/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.plugin.bugtracker.gitlab.caching;

import gitlabbt.org.gitlab4j.api.models.AbstractUser;
import gitlabbt.org.gitlab4j.api.models.Epic;
import gitlabbt.org.gitlab4j.api.models.Label;
import gitlabbt.org.gitlab4j.api.models.Milestone;
import gitlabbt.org.gitlab4j.api.models.ProjectUser;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.support.CronExpression;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;
import org.squashtest.csp.core.bugtracker.spi.BugTrackerCacheInfo;
import org.squashtest.tm.domain.bugtracker.BugTracker;
import org.squashtest.tm.plugin.bugtracker.gitlab.caching.BugTrackerAndPath;
import org.squashtest.tm.plugin.bugtracker.gitlab.caching.CachedValues;
import org.squashtest.tm.plugin.bugtracker.gitlab.caching.CompositeCacheKey;
import org.squashtest.tm.plugin.bugtracker.gitlab.caching.GitLabValuePool;
import org.squashtest.tm.plugin.bugtracker.gitlab.caching.ValueCacheDao;
import org.squashtest.tm.plugin.bugtracker.gitlab.caching.ValueCacheQueue;
import org.squashtest.tm.plugin.bugtracker.gitlab.caching.ValueCacheWorker;
import org.squashtest.tm.plugin.bugtracker.gitlab.exceptions.CannotFindNextMatchWithCronException;
import org.squashtest.tm.plugin.bugtracker.gitlab.exceptions.InvalidCacheUpdateCronException;
import org.squashtest.tm.service.servers.ManageableCredentials;
import org.squashtest.tm.service.servers.StoredCredentialsManager;

@Service
public class ValueCacheManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ValueCacheManager.class);
    public static final long MINUTES_TO_WAIT_IF_RATE_LIMIT_IS_REACHED = 5L;
    private static final long INITIAL_CACHE_CONSTRUCTION_THRESHOLD = 5L;
    private static final long MIN_CACHE_UPDATE_DELAY = 60L;
    final Map<CompositeCacheKey, CachedValues> cacheMap = new ConcurrentHashMap<CompositeCacheKey, CachedValues>();
    private final ValueCacheDao valueCacheDao;
    private final TaskScheduler taskScheduler;
    private final StoredCredentialsManager storedCredentialsManager;
    private final ValueCacheQueue updateQueue = new ValueCacheQueue();
    private final List<ValueCacheWorker> workers = new ArrayList<ValueCacheWorker>();
    private final GitLabValuePool<Label, Integer> labelPool = new GitLabValuePool<Label, Integer>(Label::getId);
    private final GitLabValuePool<Epic, Integer> epicPool = new GitLabValuePool<Epic, Integer>(Epic::getId);
    private final GitLabValuePool<Milestone, Integer> milestonePool = new GitLabValuePool<Milestone, Integer>(Milestone::getId);
    private final GitLabValuePool<ProjectUser, Integer> userPool = new GitLabValuePool<ProjectUser, Integer>(AbstractUser::getId);
    @Value(value="${squash.bugtracker.cache-update-delay:86400}")
    private long cacheUpdateDelay;
    @Value(value="${squash.bugtracker.cache-update-cron:#{null}}")
    private String cacheUpdateCron;
    @Value(value="${squash.bugtracker.cache-worker-pool-size:5}")
    private int workerPoolSize = 5;

    public ValueCacheManager(ValueCacheDao valueCacheDao, TaskScheduler taskScheduler, StoredCredentialsManager storedCredentialsManager) {
        this.valueCacheDao = valueCacheDao;
        this.taskScheduler = taskScheduler;
        this.storedCredentialsManager = storedCredentialsManager;
    }

    @PostConstruct
    public void init() {
        this.initializeWorkerPool();
        this.startBackgroundThread();
    }

    public boolean isCacheEnabled(long bugTrackerId) {
        ManageableCredentials manageableCredentials = this.storedCredentialsManager.findReportingCacheCredentials(bugTrackerId);
        return manageableCredentials != null;
    }

    private void initializeWorkerPool() {
        if (this.workerPoolSize < 1) {
            throw new IllegalArgumentException("Worker pool size (defined with property \"squash.bugtracker.cache-worker-pool-size\") must be at least 1.");
        }
        for (int i = 0; i < this.workerPoolSize; ++i) {
            this.workers.add(new ValueCacheWorker(this, i));
        }
    }

    private void startBackgroundThread() {
        if (this.cacheUpdateCron != null) {
            this.scheduleCronBasedCacheUpdate();
        } else {
            this.scheduleFixedRateCacheUpdate();
        }
    }

    private void scheduleFixedRateCacheUpdate() {
        long configuredCacheUpdateDelayInMs = this.cacheUpdateDelay * 1000L;
        long cacheUpdateDelayInMs = Math.max(configuredCacheUpdateDelayInMs, 60000L);
        this.taskScheduler.scheduleAtFixedRate(this::refresh, cacheUpdateDelayInMs);
    }

    private void scheduleCronBasedCacheUpdate() {
        long minutesUntilNextUpdate = this.getMinutesUntilNextUpdate();
        if (minutesUntilNextUpdate > 5L) {
            LOGGER.info("Found cron expression with value \"{}\". The initial cache construction will begin now, and the next cache update is in {} minutes.", (Object)this.cacheUpdateCron, (Object)minutesUntilNextUpdate);
            this.taskScheduler.schedule(this::refresh, Instant.now());
        } else {
            LOGGER.info("Found cron expression with value \"{}\". The initial cache construction will begin in {} minutes.", (Object)this.cacheUpdateCron, (Object)minutesUntilNextUpdate);
        }
        this.taskScheduler.schedule(this::refresh, (Trigger)new CronTrigger(this.cacheUpdateCron));
    }

    private long getMinutesUntilNextUpdate() {
        if (!CronExpression.isValidExpression((String)this.cacheUpdateCron)) {
            throw new InvalidCacheUpdateCronException(this.cacheUpdateCron);
        }
        CronExpression expression = CronExpression.parse((String)this.cacheUpdateCron);
        Temporal next = expression.next((Temporal)LocalDateTime.now());
        if (next == null) {
            throw new CannotFindNextMatchWithCronException(this.cacheUpdateCron);
        }
        return LocalDateTime.now().until(next, ChronoUnit.MINUTES);
    }

    private void refresh() {
        this.evictUnusedPoolItems();
        LOGGER.info("Start cache refresh.");
        this.buildCache();
        LOGGER.info("Done queuing cache refresh instructions.");
    }

    private void evictUnusedPoolItems() {
        LOGGER.trace("Evicting unused items from value pools");
        long poolSizeBeforeEviction = this.labelPool.size() + this.epicPool.size() + this.milestonePool.size() + this.userPool.size();
        long valuePoolEvictionDelay = this.cacheUpdateDelay * 3L * 1000L;
        this.labelPool.evictUnusedItems(valuePoolEvictionDelay);
        this.epicPool.evictUnusedItems(valuePoolEvictionDelay);
        this.milestonePool.evictUnusedItems(valuePoolEvictionDelay);
        this.userPool.evictUnusedItems(valuePoolEvictionDelay);
        int poolSizeAfterEviction = this.labelPool.size() + this.epicPool.size() + this.milestonePool.size() + this.userPool.size();
        LOGGER.trace("Removed {} out of {} items from value pools", (Object)(poolSizeBeforeEviction - (long)poolSizeAfterEviction), (Object)poolSizeBeforeEviction);
    }

    private void buildCache() {
        Map<Long, List<String>> projectPathsByBugTrackerId = this.valueCacheDao.getProjectPathsByBugTrackerId();
        List<BugTracker> bugTrackers = this.valueCacheDao.getAllGitLabBugTrackers();
        ArrayList keysStillInUse = new ArrayList();
        bugTrackers.forEach(bugTracker -> {
            HashSet projectPaths = new HashSet(projectPathsByBugTrackerId.getOrDefault(bugTracker.getId(), List.of()));
            projectPaths.forEach(projectPath -> {
                this.addToQueue((BugTracker)bugTracker, (String)projectPath);
                keysStillInUse.add(new CompositeCacheKey(bugTracker.getId(), (String)projectPath));
            });
        });
        this.cacheMap.keySet().removeIf(key -> !keysStillInUse.contains(key));
        this.processQueue();
    }

    public Optional<CachedValues> findCachedValues(BugTracker bugTracker, String path) {
        Long bugtrackerId = bugTracker.getId();
        CompositeCacheKey key = new CompositeCacheKey(bugtrackerId, path);
        CachedValues cachedValues = this.cacheMap.get(key);
        if (cachedValues == null || !cachedValues.hasCachedValues()) {
            this.addToQueueIfNeeded(bugTracker, path);
            return Optional.empty();
        }
        return Optional.of(cachedValues);
    }

    public void refreshCache(BugTracker bugTracker, long projectId) {
        this.taskScheduler.schedule(() -> this.refreshForProject(bugTracker, projectId), new Date());
    }

    private void refreshForProject(BugTracker bugTracker, long projectId) {
        LOGGER.info("Refreshing cache for project {} and bugtracker {}", (Object)projectId, (Object)bugTracker.getId());
        Set<String> projectPaths = this.valueCacheDao.getGitLabProjectPathsByTmProjectId(projectId);
        for (String projectPath : projectPaths) {
            this.addToQueueIfNeeded(bugTracker, projectPath);
        }
        this.processQueue();
    }

    public void refreshCache(BugTracker bugTracker) {
        this.taskScheduler.schedule(() -> this.refreshForBugTracker(bugTracker), new Date());
    }

    private void refreshForBugTracker(BugTracker bugTracker) {
        LOGGER.info("Refreshing cache for bugtracker {}", (Object)bugTracker.getId());
        Set<String> projectPaths = this.valueCacheDao.getGitLabProjectPathsByBugTrackerId(bugTracker.getId());
        for (String projectPath : projectPaths) {
            this.addToQueue(bugTracker, projectPath);
        }
        this.processQueue();
    }

    public boolean hasCacheError(Long id, String projectPath) {
        CompositeCacheKey key = new CompositeCacheKey(id, projectPath);
        CachedValues cachedValues = this.cacheMap.get(key);
        if (cachedValues == null) {
            return false;
        }
        return cachedValues.hasCacheError();
    }

    public BugTrackerCacheInfo getCacheInfo() {
        BugTrackerCacheInfo cacheInfo = new BugTrackerCacheInfo();
        cacheInfo.setEntries(this.cacheMap.entrySet().stream().map(entry -> {
            CompositeCacheKey key = (CompositeCacheKey)entry.getKey();
            CachedValues values = (CachedValues)entry.getValue();
            return new BugTrackerCacheInfo.Entry(key.bugTrackerId(), key.path(), values.lastUpdated(), values.lastSuccessfulUpdated(), values.hasCacheError());
        }).toList());
        return cacheInfo;
    }

    private void addToQueue(BugTracker bugTracker, String projectPath) {
        BugTrackerAndPath bugTrackerAndPath = new BugTrackerAndPath(bugTracker, projectPath);
        if (!this.updateQueue.contains(bugTrackerAndPath)) {
            LOGGER.trace("Add project {} and bug tracker {} to cache refresh queue", (Object)projectPath, (Object)bugTracker.getId());
            this.updateQueue.enqueue(bugTrackerAndPath, this.cacheMap);
        }
    }

    private void addToQueueIfNeeded(BugTracker bugTracker, String path) {
        CompositeCacheKey key = new CompositeCacheKey(bugTracker.getId(), path);
        if (this.cacheMap.containsKey(key) && !this.cacheMap.get(key).hasCacheError()) {
            LOGGER.trace("Cache for project {} and bug tracker {} is already up to date", (Object)path, (Object)bugTracker.getId());
            return;
        }
        LOGGER.trace("Queuing cache updates for project {} and bug tracker {}", (Object)path, (Object)bugTracker.getId());
        this.addToQueue(bugTracker, path);
    }

    private void processQueue() {
        this.workers.forEach(ValueCacheWorker::start);
    }

    StoredCredentialsManager getStoredCredentialsManager() {
        return this.storedCredentialsManager;
    }

    ValueCacheQueue getQueue() {
        return this.updateQueue;
    }

    TaskScheduler getTaskScheduler() {
        return this.taskScheduler;
    }

    GitLabValuePool<Label, Integer> getLabelPool() {
        return this.labelPool;
    }

    GitLabValuePool<Epic, Integer> getEpicPool() {
        return this.epicPool;
    }

    GitLabValuePool<Milestone, Integer> getMilestonePool() {
        return this.milestonePool;
    }

    GitLabValuePool<ProjectUser, Integer> getUserPool() {
        return this.userPool;
    }

    public void retryLater(BugTracker bugTracker, String path) {
        this.getTaskScheduler().schedule(() -> {
            this.addToQueueIfNeeded(bugTracker, path);
            this.processQueue();
        }, Instant.now().plus(5L, ChronoUnit.MINUTES));
    }
}

