/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.plugin.bugtracker.redmine3.caching;

import com.taskadapter.redmineapi.RedmineException;
import com.taskadapter.redmineapi.bean.Project;
import jakarta.annotation.PostConstruct;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.support.CronExpression;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;
import org.squashtest.csp.core.bugtracker.spi.BugTrackerCacheInfo;
import org.squashtest.tm.domain.bugtracker.BugTracker;
import org.squashtest.tm.plugin.bugtracker.redmine3.Redmine3Client;
import org.squashtest.tm.plugin.bugtracker.redmine3.RedmineClientInitializer;
import org.squashtest.tm.plugin.bugtracker.redmine3.caching.RedmineBugTrackerAndProject;
import org.squashtest.tm.plugin.bugtracker.redmine3.caching.RedmineCachedValues;
import org.squashtest.tm.plugin.bugtracker.redmine3.caching.RedmineCompositeCacheKey;
import org.squashtest.tm.plugin.bugtracker.redmine3.caching.RedmineValueCacheDao;
import org.squashtest.tm.plugin.bugtracker.redmine3.caching.RedmineValueCacheQueue;
import org.squashtest.tm.plugin.bugtracker.redmine3.caching.RedmineValueCacheWorker;
import org.squashtest.tm.plugin.bugtracker.redmine3.converter.EntityConverter;
import org.squashtest.tm.plugin.bugtracker.redmine3.exception.CannotFindNextMatchWithCronException;
import org.squashtest.tm.plugin.bugtracker.redmine3.exception.InvalidCacheUpdateCronException;
import org.squashtest.tm.service.servers.ManageableCredentials;
import org.squashtest.tm.service.servers.StoredCredentialsManager;

@Service
public class RedmineValueCacheManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(RedmineValueCacheManager.class);
    private static final long INITIAL_CACHE_CONSTRUCTION_THRESHOLD = 5L;
    private static final long MIN_CACHE_UPDATE_DELAY = 60L;
    private static final String PROJECT_NOT_FOUND = "Project %s not found";
    private final StoredCredentialsManager storedCredentialsManager;
    private final TaskScheduler taskScheduler;
    private final List<RedmineValueCacheWorker> workers = new ArrayList<RedmineValueCacheWorker>();
    private final RedmineValueCacheQueue updateQueue;
    private final EntityConverter entityConverter;
    private Map<RedmineCompositeCacheKey, RedmineCachedValues> cachedValues = new ConcurrentHashMap<RedmineCompositeCacheKey, RedmineCachedValues>();
    private RedmineValueCacheDao redmineValueCacheDao;
    private RedmineClientInitializer redmineClientInitializer;
    @Value(value="${squash.bugtracker.cache-update-delay:86400}")
    private long cacheUpdateDelay;
    @Value(value="${squash.bugtracker.cache-update-cron:#{null}}")
    private String cacheUpdateCron;
    @Value(value="${squash.bugtracker.cache-worker-pool-size:5}")
    private int workerPoolSize = 5;

    public RedmineValueCacheManager(StoredCredentialsManager storedCredentialsManager, TaskScheduler taskScheduler, RedmineValueCacheDao redmineValueCacheDao, RedmineClientInitializer redmineClientInitializer, EntityConverter entityConverter) {
        this.storedCredentialsManager = storedCredentialsManager;
        this.taskScheduler = taskScheduler;
        this.redmineValueCacheDao = redmineValueCacheDao;
        this.redmineClientInitializer = redmineClientInitializer;
        this.updateQueue = new RedmineValueCacheQueue(this.cachedValues);
        this.entityConverter = entityConverter;
    }

    @PostConstruct
    public void init() {
        this.initializeWorkerPool();
        this.startBackgroundThread();
    }

    private void initializeWorkerPool() {
        if (this.workerPoolSize < 1) {
            throw new IllegalArgumentException("Worker pool size (defined with property \"squash.bugtracker.cache-worker-pool-size\") must be at least 1.");
        }
        int i = 0;
        while (i < this.workerPoolSize) {
            RedmineValueCacheWorker worker = new RedmineValueCacheWorker(this.redmineClientInitializer, this, this.entityConverter);
            worker.setId(i);
            this.workers.add(worker);
            ++i;
        }
    }

    private void startBackgroundThread() {
        if (this.cacheUpdateCron != null) {
            this.scheduleCronBasedCacheUpdate();
        } else {
            this.scheduleFixedRateCacheUpdate();
        }
    }

    private void scheduleCronBasedCacheUpdate() {
        long minutesUntilNextUpdate = this.getMinutesUntilNextUpdate();
        if (minutesUntilNextUpdate > 5L) {
            LOGGER.info("Found cron expression with value \"{}\". The initial cache construction will begin now, and the next cache update is in {} minutes.", (Object)this.cacheUpdateCron, (Object)minutesUntilNextUpdate);
            this.taskScheduler.schedule(this::refresh, Instant.now());
        } else {
            LOGGER.info("Found cron expression with value \"{}\". The initial cache construction will begin in {} minutes.", (Object)this.cacheUpdateCron, (Object)minutesUntilNextUpdate);
        }
        this.taskScheduler.schedule(this::refresh, (Trigger)new CronTrigger(this.cacheUpdateCron));
    }

    private void scheduleFixedRateCacheUpdate() {
        long configuredCacheUpdateDelayInMs = this.cacheUpdateDelay * 1000L;
        long cacheUpdateDelayInMs = Math.max(configuredCacheUpdateDelayInMs, 60000L);
        this.taskScheduler.scheduleAtFixedRate(this::refresh, Duration.ofMillis(cacheUpdateDelayInMs));
    }

    private long getMinutesUntilNextUpdate() {
        if (!CronExpression.isValidExpression((String)this.cacheUpdateCron)) {
            throw new InvalidCacheUpdateCronException(this.cacheUpdateCron);
        }
        CronExpression expression = CronExpression.parse((String)this.cacheUpdateCron);
        Temporal next = expression.next((Temporal)LocalDateTime.now());
        if (next == null) {
            throw new CannotFindNextMatchWithCronException(this.cacheUpdateCron);
        }
        return LocalDateTime.now().until(next, ChronoUnit.MINUTES);
    }

    private void refresh() {
        LOGGER.info("Start cache refresh.");
        this.buildCache();
        LOGGER.info("Done queuing cache refresh instructions.");
    }

    private void buildCache() {
        List<BugTracker> bugTrackers = this.redmineValueCacheDao.getAllRedmineBugTrackers();
        Set keysStillInUse = bugTrackers.stream().map(this::refreshCacheForBugTracker).flatMap(Collection::stream).collect(Collectors.toSet());
        this.cachedValues.keySet().removeIf(key -> !keysStillInUse.contains(key));
        this.processQueue();
    }

    private Set<RedmineCompositeCacheKey> refreshCacheForBugTracker(BugTracker bugTracker) {
        List<Project> redmineProjects;
        HashSet<RedmineCompositeCacheKey> keysStillInUse = new HashSet<RedmineCompositeCacheKey>();
        Redmine3Client client = this.initRedmineClient(bugTracker);
        if (client == null) {
            LOGGER.warn("No client for bug tracker {}, skipping", (Object)bugTracker.getId());
            return keysStillInUse;
        }
        Set<String> projectPaths = this.redmineValueCacheDao.getRedmineProjectPathsByBugTrackerId(bugTracker.getId());
        try {
            redmineProjects = client.findProjects(projectPaths);
        }
        catch (Exception e) {
            LOGGER.error("Error fetching projects for bug tracker {}", (Object)bugTracker.getId(), (Object)e);
            projectPaths.forEach(projectPath -> {
                RedmineCompositeCacheKey key = new RedmineCompositeCacheKey(bugTracker.getId(), (String)projectPath);
                this.handleCacheError(key, new RedmineException(PROJECT_NOT_FOUND.formatted(projectPath)), (String)projectPath, bugTracker);
                keysStillInUse.add(key);
            });
            return keysStillInUse;
        }
        projectPaths.forEach(projectPath -> {
            Project redmineProject = redmineProjects.stream().filter(p -> p.getName().equals(projectPath)).findFirst().orElse(null);
            RedmineCompositeCacheKey cacheKey = new RedmineCompositeCacheKey(bugTracker.getId(), (String)projectPath);
            if (redmineProject != null) {
                LOGGER.trace("Queuing cache updates for project {} and bug tracker {}", (Object)redmineProject.getName(), (Object)bugTracker.getId());
                this.addToQueue(bugTracker, redmineProject);
            } else {
                this.handleCacheError(cacheKey, new RedmineException(PROJECT_NOT_FOUND.formatted(projectPath)), (String)projectPath, bugTracker);
            }
            keysStillInUse.add(cacheKey);
        });
        return keysStillInUse;
    }

    private void addToQueue(BugTracker bugTracker, Project project) {
        RedmineBugTrackerAndProject redmineBugTrackerAndProject = new RedmineBugTrackerAndProject(bugTracker, project);
        if (!this.updateQueue.contains(redmineBugTrackerAndProject)) {
            LOGGER.trace("Add project {} and bug tracker {} to cache refresh queue", (Object)project.getName(), (Object)bugTracker.getId());
            this.updateQueue.enqueue(redmineBugTrackerAndProject);
        }
    }

    private void addToQueueIfNeeded(BugTracker bugTracker, Set<String> projectPaths) {
        Set<String> pathsToUpdate = this.computePathsToUpdate(bugTracker, projectPaths);
        if (pathsToUpdate.isEmpty()) {
            return;
        }
        Redmine3Client client = this.initRedmineClient(bugTracker);
        if (client == null) {
            LOGGER.warn("No client for bug tracker {}, skipping", (Object)bugTracker.getId());
            return;
        }
        List<Project> redmineProjects = client.findProjects(pathsToUpdate);
        pathsToUpdate.forEach(pathToUpdate -> {
            Project redmineProject = redmineProjects.stream().filter(p -> p.getName().equals(pathToUpdate)).findFirst().orElse(null);
            RedmineCompositeCacheKey cacheKey = new RedmineCompositeCacheKey(bugTracker.getId(), (String)pathToUpdate);
            if (redmineProject != null) {
                LOGGER.trace("Queuing cache updates for project {} and bug tracker {}", (Object)redmineProject.getName(), (Object)bugTracker.getId());
                this.addToQueue(bugTracker, redmineProject);
            } else {
                this.handleCacheError(cacheKey, new RedmineException(PROJECT_NOT_FOUND.formatted(pathToUpdate)), (String)pathToUpdate, bugTracker);
            }
        });
    }

    private Set<String> computePathsToUpdate(BugTracker bugTracker, Set<String> projectPaths) {
        return projectPaths.stream().filter(path -> !this.isUpToDate(bugTracker, (String)path)).collect(Collectors.toSet());
    }

    private boolean isUpToDate(BugTracker bugTracker, String path) {
        boolean upToDate;
        RedmineCompositeCacheKey key = new RedmineCompositeCacheKey(bugTracker.getId(), path);
        boolean bl = upToDate = this.cachedValues.containsKey(key) && !this.cachedValues.get(key).hasCacheError();
        if (upToDate) {
            LOGGER.trace("Cache for project {} and bug tracker {} is already up to date", (Object)path, (Object)bugTracker.getId());
        }
        return upToDate;
    }

    private void processQueue() {
        this.workers.forEach(RedmineValueCacheWorker::start);
    }

    public boolean isCacheEnabled(long bugTrackerId) {
        ManageableCredentials manageableCredentials = this.storedCredentialsManager.findReportingCacheCredentials(bugTrackerId);
        return manageableCredentials != null;
    }

    TaskScheduler getTaskScheduler() {
        return this.taskScheduler;
    }

    public void refreshCache(BugTracker bugTracker) {
        this.taskScheduler.schedule(() -> {
            this.refreshCacheForBugTracker(bugTracker);
            this.processQueue();
        }, Instant.now().plusSeconds(1L));
    }

    public void refreshCache(BugTracker bugTracker, Long projectId) {
        this.taskScheduler.schedule(() -> this.refreshForProject(bugTracker, projectId), Instant.now());
    }

    private void refreshForProject(BugTracker bugTracker, long projectId) {
        LOGGER.info("Refreshing cache for project {} and bugtracker {}", (Object)projectId, (Object)bugTracker.getId());
        Set<String> projectPaths = this.redmineValueCacheDao.getRedmineProjectPathsByTmProjectId(projectId);
        this.addToQueueIfNeeded(bugTracker, projectPaths);
        this.processQueue();
    }

    RedmineValueCacheQueue getQueue() {
        return this.updateQueue;
    }

    private Redmine3Client initRedmineClient(BugTracker bugTracker) {
        return this.redmineClientInitializer.initRedmineClient(bugTracker);
    }

    public BugTrackerCacheInfo getCacheInfo() {
        BugTrackerCacheInfo cacheInfo = new BugTrackerCacheInfo();
        cacheInfo.setEntries(this.cachedValues.entrySet().stream().map(entry -> {
            RedmineCompositeCacheKey key = (RedmineCompositeCacheKey)entry.getKey();
            RedmineCachedValues values = (RedmineCachedValues)entry.getValue();
            return new BugTrackerCacheInfo.Entry(Long.valueOf(key.bugTrackerId()), key.projectPath(), values.lastUpdated(), values.lastSuccessfulUpdated(), values.hasCacheError());
        }).toList());
        return cacheInfo;
    }

    public boolean hasLastCacheUpdateFailed(Long bugTrackerId, String projectPath) {
        RedmineCompositeCacheKey key = new RedmineCompositeCacheKey(bugTrackerId, projectPath);
        RedmineCachedValues cache = this.cachedValues.get(key);
        if (cache == null) {
            return false;
        }
        return cache.hasCacheError() && cache.withLastUpdateError().lastSuccessfulUpdated() != null;
    }

    public boolean hasCacheError(Long bugTrackerId, String projectPath) {
        return Optional.ofNullable(this.getCachedValues()).map(map -> (RedmineCachedValues)map.get(new RedmineCompositeCacheKey(bugTrackerId, projectPath))).map(RedmineCachedValues::hasCacheError).orElse(false);
    }

    public void updateCache(RedmineCompositeCacheKey key, RedmineCachedValues values) {
        this.cachedValues.put(key, values);
    }

    public void handleCacheError(RedmineCompositeCacheKey key, RedmineException exception, String projectName, BugTracker bugTracker) {
        if (this.cachedValues.containsKey(key)) {
            RedmineCachedValues updated = this.cachedValues.get(key).withLastUpdateError();
            this.cachedValues.put(key, updated);
        } else {
            this.cachedValues.put(key, RedmineCachedValues.error());
        }
        Object message = "Error while updating cache for project %s and bug tracker %s.".formatted(projectName, bugTracker.getId());
        if (this.cachedValues.get(key).hasCachedValues()) {
            message = (String)message + " The previously cached values will be used until the next update.";
        }
        LOGGER.error((String)message, (Throwable)exception);
    }

    public Map<RedmineCompositeCacheKey, RedmineCachedValues> getCachedValues() {
        return Map.copyOf(this.cachedValues);
    }
}

