/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (sync-xsquash4jira) project.
 *     Copyright (C) 2017 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
define(["./BaseModel", "./BaseScreen", "./IterationFormModel", "squash.configmanager", "custom-field-values", "squash.translator"], 
		function(BaseModel, BaseScreen, IterationFormModel, confman, cufsManager, translator){
	
	
	var dataService = squashtm.app.contextRoot + 'jirasync';
	
	
	
	/*
	 * Here we define a hook that make CKEditors trigger a change when they actually change
	 * more specifically, the textarea they refer to will trigger the event, not them.
	 */
	
	CKEDITOR.on('instanceReady', function(evt){
		evt.editor.on('blur', function(evt){
			var thiseditor = evt.editor;
			var txtarea = thiseditor.element.$;
			$(txtarea).trigger('change');
		});
	});
	
	
	/*
	 * Fix the layout z-index of datepickers : because the main pane has a lower index than the
	 * rest of the interface, part of the datepickers stay hidden behind other parts of the UI.
	 * We must ensure that it won't be the case.
	 * 
	 *  There are no suitable hook that allow to do that via the api so overriding the 
	 *  method is the only solution
	 */ 
	var oldpicker = $.datepicker._showDatepicker;
	$.datepicker._showDatepicker = function(input){
		oldpicker.apply(this, arguments);
		var inst = $.datepicker._getInst(input.target || input);
		inst.dpDiv.zIndex(1000);
	}
	
	// ********* iteration creation ***************
	
	var IterationModel = BaseModel.extend({
		
		consumes : ['selectedNode'],
		produces : ['iteration'],
		
		defaults : {
			//from megamodel
			selectedNode : null,
			iteration : new IterationFormModel()
		},
			
		// must return a promise on any cases
		fetchMetaIfNeeded : function(){
			if (this.isFresh()){
				return $.Deferred().resolve().promise();
			}
			
			var url = dataService + '/campaigns/'+this.selectedNode().entityId + '/iterations/meta';
			var self = this;
			return $.getJSON(url).done(function(json){				
				self.configureIterationModel(json);
				self.fresh = true;
				return this;
			});
		},
		
		configureIterationModel : function(json){
			var iter = this.iteration();
			
			// initialize the customfields of the iteration
			var jsonCfs = json.customFields || [];
			var iterCfs = {};
			jsonCfs.forEach(function(cf){
				iterCfs[cf.id] = cf.defaultValue;
				iter.customFields[cf.id] = cf.defaultValue;
			});
			iter.customFields(iterCfs);
			
			// also set the metadata
			iter.meta(json);
			
			this.iteration(iter);
		}
		
	});
	

	
	// **** Iteration (optional) : available only if strategy is "CREATE_IT" ******
	
	var IterationScreen = BaseScreen.extend({
		
		name : "iteration-create",
		
		el : "#screen-iteration-create",
		
		template : "#jirsync-plan-iteration",
			
		cufsHandler : null,
		
		render : function(){
			var self = this;
			
			this.__renderWait();
			this.model
				.fetchMetaIfNeeded()
				.done(function(){
					self.renderPane();					
			});
			
			return this;
			
		},
		
		templateModel : function(){
			return this.model.iteration().attributes;
		},
		
		renderPane : function(){
			BaseScreen.prototype.render.call(this);

			// base attributes additional init
			var richconf = confman.getStdCkeditor();
			this.description().ckeditor(function(){}, richconf);
			
			// init the customfields
			this._initCustomfields();
			
			// needs to reset the values of the custom fields from the model
			this._initCustomfieldValues();
			
			// also need to add a hook for change events for all tag customfields
			this._initTagsChangeHook();
			
			
		},
		
		_initCustomfields : function(){
			
			var cufDefs = this.model.iteration().meta().customFields;
			
			this.cufsHandler = cufsManager.newCreationPopupCUFHandler({table : $("#iteration-creation-table")});
			this.cufsHandler.loadPanel(cufDefs);
		},
		
		_initCustomfieldValues : function(){
			var cfValues = this.model.iteration().customFields;			
			
			// needs to reset the values of the custom fields from the model
			this.$el.find(".create-node-custom-field").each(function(){
				var cf = $(this),
					cfId = cf.data('cuf-id');
				var val = cfValues[cfId];
				if (!!val && val.length > 0){
					cf.editableCustomfield("value", val);
				}
			});
		},
		
		_initTagsChangeHook : function(){
			
			function wrapHandler(handler){
				return function(){
					if (!!handler){
						handler.apply(this, arguments);
					}
					$(this).trigger('change');
				}
			}
			
			// wrap the previous 'afterTagAdded' and 'afterTagRemoved' handlers if any
			this.$el.find('.squash-tagit').each(function(){
				var $this = $(this);
				var options = $this.data('squashTagit').options; 
				
				var newOnAdded = wrapHandler(options.afterTagAdded);
				var newOnRemove = wrapHandler(options.afterTagRemoved);
				
				$this.squashTagit('option', 'afterTagAdded', newOnAdded);
				$this.squashTagit('option', 'afterTagRemoved', newOnRemove);
			});
		},
		
		
		_destroyWidgets : function(){
			var desc = this.description();
			if (desc.length> 0){
				confman.destroyCkeditor(this.description());
			}
			$(".create-node-custom-field").each(function(){
				$(this).editableCustomfield("destroy");
			});
		},
		
		
		// ************** events ****************
		
		/*
		 * Remember : the CKEditors here trigger a change only thanks to the 
		 * fix at the head of the file
		 */
		events : {
			"change #iter-name" 				: 	"setName",
			"change #iter-ref"					: 	"setReference",
			"change #iter-desc" 				: 	"setDescription",
			"change .create-node-custom-field" 	: "setCustomfield"
		},
		
		isComplete : function(){
			return (this.model.iteration().validate() === undefined);
		},
		
		setName : function(){
			var name = $("#iter-name").val();
			this._updateAttribute({name : name});
		},
		
		setReference : function(){
			var ref = $("#iter-ref").val();
			this._updateAttribute({reference : ref});
		},
		
		setDescription : function(){
			var desc = CKEDITOR.instances["iter-desc"].getData();
			this._updateAttribute({description : desc});
		},
		
		setCustomfield : function(evt){
			var cuf = $(evt.currentTarget);
			var val = cuf.editableCustomfield('value');
			var cfId = cuf.data('cuf-id');
			
			var cufs = this.model.iteration().customFields();
			cufs[cfId] = val;
			
			this._updateAttribute({customFields : cufs});
		},
		
		_updateAttribute : function(attr){
			this.resetErrors();
			var iter = this.model.iteration();
			iter.set(attr);
			if (! iter.isValid()){
				this.showErrors();
			}
			else{
				this.model.iteration(iter);
			}
		},
		
		// *************** misc *****************
		
		resetErrors : function(){
			this.$el.find('.error-message').text('');
		},
		
		showErrors : function(){
			var codes = IterationFormModel.prototype.errorCodes;			
			var errors = this.model.iteration().validationError;
			
			for (var prop in errors){
			
				var field = prop;
				var msg;
				switch(errors[prop]){
					case codes.NOT_BLANK : msg = translator.get('henix.jirasync.execplan.designer.validation.notblank'); break;
					case codes.DUPLICATE : msg = translator.get('henix.jirasync.execplan.designer.validation.duplicate'); break;
					case codes.INVALID_FORMAT : msg = translator.get('henix.jirasync.execplan.designer.validation.invalid'); break;
					case codes.NOT_A_NUMBER : msg = translator.get('henix.jirasync.execplan.designer.validation.nan'); break;
					default : throw "unknown error code : "+errors[prop];
				}
				
				this.$el.find('.error-message.'+prop+'-error').text(msg);
			}
		},
		
		description : function(){
			return $("#iter-desc");
		}	
		
	});
	
	
	return {
		model : IterationModel,
		screen : IterationScreen
	};
	
});