/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching;

import jakarta.annotation.PostConstruct;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.support.CronExpression;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;
import org.squashtest.csp.core.bugtracker.spi.BugTrackerCacheInfo;
import org.squashtest.tm.domain.bugtracker.BugTracker;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching.AzureDevopsBugTrackerAndPath;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching.AzureDevopsCachedValues;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching.AzureDevopsCompositeCacheKey;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching.AzureDevopsValueCacheDao;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching.AzureDevopsValueCacheQueue;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching.AzureDevopsValueCacheWorker;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.caching.AzureDevopsValuePool;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.client.AzureDevOpsClientFactory;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.client.model.Nodes;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.client.model.Tag;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.client.model.TeamMember;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.exception.CannotFindNextMatchWithCronException;
import org.squashtest.tm.plugin.bugtracker.azuredevops.internal.exception.InvalidCacheUpdateCronException;
import org.squashtest.tm.service.servers.ManageableCredentials;
import org.squashtest.tm.service.servers.StoredCredentialsManager;

@Service
public class AzureDevopsValueCacheManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(AzureDevopsValueCacheManager.class);
    private static final long INITIAL_CACHE_CONSTRUCTION_THRESHOLD = 5L;
    private static final long MIN_CACHE_UPDATE_DELAY = 60L;
    final Map<AzureDevopsCompositeCacheKey, AzureDevopsCachedValues> cacheMap = new ConcurrentHashMap<AzureDevopsCompositeCacheKey, AzureDevopsCachedValues>();
    private final AzureDevopsValueCacheDao azureDevopsValueCacheDao;
    private final TaskScheduler taskScheduler;
    private final StoredCredentialsManager storedCredentialsManager;
    private final AzureDevOpsClientFactory azureDevOpsClientFactory;
    private final AzureDevopsValueCacheQueue updateQueue = new AzureDevopsValueCacheQueue();
    private final List<AzureDevopsValueCacheWorker> workers = new ArrayList<AzureDevopsValueCacheWorker>();
    private final AzureDevopsValuePool<TeamMember, String> teamMembersPool = new AzureDevopsValuePool<TeamMember, String>(teamMember -> teamMember.getIdentity().getUniqueName());
    private final AzureDevopsValuePool<Tag, String> tagsPool = new AzureDevopsValuePool<Tag, String>(Tag::getId);
    private final AzureDevopsValuePool<Nodes, String> areasPool = new AzureDevopsValuePool<Nodes, String>(Nodes::getId);
    private final AzureDevopsValuePool<Nodes, String> iterationsPool = new AzureDevopsValuePool<Nodes, String>(Nodes::getId);
    @Value(value="${squash.bugtracker.cache-update-delay:86400}")
    private long cacheUpdateDelay;
    @Value(value="${squash.bugtracker.cache-update-cron:#{null}}")
    private String cacheUpdateCron;
    @Value(value="${squash.bugtracker.cache-worker-pool-size:5}")
    private int workerPoolSize = 5;

    public AzureDevopsValueCacheManager(AzureDevopsValueCacheDao azureDevopsValueCacheDao, TaskScheduler taskScheduler, StoredCredentialsManager storedCredentialsManager, AzureDevOpsClientFactory azureDevOpsClientFactory) {
        this.azureDevopsValueCacheDao = azureDevopsValueCacheDao;
        this.taskScheduler = taskScheduler;
        this.storedCredentialsManager = storedCredentialsManager;
        this.azureDevOpsClientFactory = azureDevOpsClientFactory;
    }

    @PostConstruct
    public void init() {
        this.initializeWorkerPool();
        this.startBackgroundThread();
    }

    private void initializeWorkerPool() {
        if (this.workerPoolSize < 1) {
            throw new IllegalArgumentException("Worker pool size (defined with property \"squash.bugtracker.cache-worker-pool-size\") must be at least 1.");
        }
        for (int i = 0; i < this.workerPoolSize; ++i) {
            this.workers.add(new AzureDevopsValueCacheWorker(this, this.azureDevOpsClientFactory, i));
        }
    }

    private void startBackgroundThread() {
        if (this.cacheUpdateCron != null) {
            this.scheduleCronBasedCacheUpdate();
        } else {
            this.scheduleFixedRateCacheUpdate();
        }
    }

    private void scheduleFixedRateCacheUpdate() {
        long configuredCacheUpdateDelayInMs = this.cacheUpdateDelay * 1000L;
        long cacheUpdateDelayInMs = Math.max(configuredCacheUpdateDelayInMs, 60000L);
        this.taskScheduler.scheduleAtFixedRate(this::refresh, cacheUpdateDelayInMs);
    }

    private void scheduleCronBasedCacheUpdate() {
        long minutesUntilNextUpdate = this.getMinutesUntilNextUpdate();
        if (minutesUntilNextUpdate > 5L) {
            LOGGER.info("Found cron expression with value \"{}\". The initial cache construction will begin now, and the next cache update is in {} minutes.", (Object)this.cacheUpdateCron, (Object)minutesUntilNextUpdate);
            this.taskScheduler.schedule(this::refresh, Instant.now());
        } else {
            LOGGER.info("Found cron expression with value \"{}\". The initial cache construction will begin in {} minutes.", (Object)this.cacheUpdateCron, (Object)minutesUntilNextUpdate);
        }
        this.taskScheduler.schedule(this::refresh, (Trigger)new CronTrigger(this.cacheUpdateCron));
    }

    private long getMinutesUntilNextUpdate() {
        if (!CronExpression.isValidExpression((String)this.cacheUpdateCron)) {
            throw new InvalidCacheUpdateCronException(this.cacheUpdateCron);
        }
        CronExpression expression = CronExpression.parse((String)this.cacheUpdateCron);
        Temporal next = expression.next((Temporal)LocalDateTime.now());
        if (next == null) {
            throw new CannotFindNextMatchWithCronException(this.cacheUpdateCron);
        }
        return LocalDateTime.now().until(next, ChronoUnit.MINUTES);
    }

    private void refresh() {
        this.evictUnusedPoolItems();
        LOGGER.info("Start cache refresh.");
        this.buildCache();
        LOGGER.info("Done queuing cache refresh instructions.");
    }

    private void evictUnusedPoolItems() {
        LOGGER.trace("Evicting unused items from value pools");
        long poolSizeBeforeEviction = this.teamMembersPool.size() + this.tagsPool.size() + this.areasPool.size() + this.iterationsPool.size();
        long valuePoolEvictionDelay = this.cacheUpdateDelay * 3L * 1000L;
        this.teamMembersPool.evictUnusedItems(valuePoolEvictionDelay);
        this.tagsPool.evictUnusedItems(valuePoolEvictionDelay);
        this.areasPool.evictUnusedItems(valuePoolEvictionDelay);
        this.iterationsPool.evictUnusedItems(valuePoolEvictionDelay);
        int poolSizeAfterEviction = this.teamMembersPool.size() + this.tagsPool.size() + this.areasPool.size() + this.iterationsPool.size();
        LOGGER.trace("Removed {} out of {} items from value pools", (Object)(poolSizeBeforeEviction - (long)poolSizeAfterEviction), (Object)poolSizeBeforeEviction);
    }

    private void buildCache() {
        Map<Long, List<String>> projectPathsByBugTrackerId = this.azureDevopsValueCacheDao.getProjectPathsByBugTrackerId();
        List<BugTracker> bugTrackers = this.azureDevopsValueCacheDao.getAllAzureDevopsBugTrackers();
        ArrayList keysStillInUse = new ArrayList();
        bugTrackers.forEach(bugTracker -> {
            HashSet projectPaths = new HashSet(projectPathsByBugTrackerId.getOrDefault(bugTracker.getId(), List.of()));
            projectPaths.forEach(projectPath -> {
                this.addToQueue((BugTracker)bugTracker, (String)projectPath);
                keysStillInUse.add(new AzureDevopsCompositeCacheKey(bugTracker.getId(), (String)projectPath));
            });
        });
        this.cacheMap.keySet().removeIf(key -> !keysStillInUse.contains(key));
        this.processQueue();
    }

    public AzureDevopsValuePool<TeamMember, String> getTeamMembersPool() {
        return this.teamMembersPool;
    }

    public AzureDevopsValuePool<Tag, String> getTagsPool() {
        return this.tagsPool;
    }

    public AzureDevopsValuePool<Nodes, String> getAreasPool() {
        return this.areasPool;
    }

    public AzureDevopsValuePool<Nodes, String> getIterationsPool() {
        return this.iterationsPool;
    }

    public boolean isCacheEnabled(long bugTrackerId) {
        ManageableCredentials manageableCredentials = this.storedCredentialsManager.findReportingCacheCredentials(bugTrackerId);
        return manageableCredentials != null;
    }

    public boolean hasLastCacheUpdateFailed(Long bugTrackerId, String projectPath) {
        AzureDevopsCompositeCacheKey key = new AzureDevopsCompositeCacheKey(bugTrackerId, projectPath);
        AzureDevopsCachedValues cachedValues = this.cacheMap.get(key);
        if (cachedValues == null) {
            return false;
        }
        return cachedValues.hasCacheError() && cachedValues.withLastUpdateError().lastSuccessfulUpdated() != null;
    }

    public Optional<AzureDevopsCachedValues> findCachedValues(BugTracker bugTracker, String path) {
        AzureDevopsCompositeCacheKey key = new AzureDevopsCompositeCacheKey(bugTracker.getId(), path);
        AzureDevopsCachedValues azureDevopsCachedValues = this.cacheMap.get(key);
        if (azureDevopsCachedValues == null || !azureDevopsCachedValues.hasCachedValues()) {
            this.addToQueueIfNeeded(bugTracker, path);
            return Optional.empty();
        }
        return Optional.of(azureDevopsCachedValues);
    }

    private void addToQueueIfNeeded(BugTracker bugTracker, String path) {
        AzureDevopsCompositeCacheKey key = new AzureDevopsCompositeCacheKey(bugTracker.getId(), path);
        if (this.cacheMap.containsKey(key) && !this.cacheMap.get(key).hasCacheError()) {
            LOGGER.trace("Cache for project {} and bug tracker {} is already up to date", (Object)path, (Object)bugTracker.getId());
            return;
        }
        LOGGER.trace("Queuing cache updates for project {} and bug tracker {}", (Object)path, (Object)bugTracker.getId());
        this.addToQueue(bugTracker, path);
    }

    private void addToQueue(BugTracker bugTracker, String projectPath) {
        AzureDevopsBugTrackerAndPath azureDevopsBugTrackerAndPath = new AzureDevopsBugTrackerAndPath(bugTracker, projectPath);
        if (!this.updateQueue.contains(azureDevopsBugTrackerAndPath)) {
            LOGGER.trace("Add project {} and bug tracker {} to cache refresh queue", (Object)projectPath, (Object)bugTracker.getId());
            this.updateQueue.enqueue(azureDevopsBugTrackerAndPath, this.cacheMap);
        }
    }

    public void refreshCache(BugTracker bugTracker) {
        this.taskScheduler.schedule(() -> this.refreshForBugtracker(bugTracker), new Date());
    }

    private void refreshForBugtracker(BugTracker bugTracker) {
        LOGGER.info("Refreshing cache for bugtracker {}", (Object)bugTracker.getId());
        Set<String> projectPaths = this.azureDevopsValueCacheDao.getAzureDevopsProjectPathsByBugTrackerId(bugTracker.getId());
        for (String projectPath : projectPaths) {
            this.addToQueue(bugTracker, projectPath);
        }
        this.processQueue();
    }

    public void refreshCache(BugTracker bugTracker, Long projectId) {
        this.taskScheduler.schedule(() -> this.refreshForProject(bugTracker, projectId), new Date());
    }

    private void refreshForProject(BugTracker bugTracker, long projectId) {
        LOGGER.info("Refreshing cache for project {} and bugtracker {}", (Object)projectId, (Object)bugTracker.getId());
        Set<String> projectPaths = this.azureDevopsValueCacheDao.getAzureDevopsProjectPathsByTmProjectId(projectId);
        for (String projectPath : projectPaths) {
            this.addToQueueIfNeeded(bugTracker, projectPath);
        }
        this.processQueue();
    }

    private void processQueue() {
        this.workers.forEach(AzureDevopsValueCacheWorker::start);
    }

    public BugTrackerCacheInfo getCacheInfo() {
        BugTrackerCacheInfo cacheInfo = new BugTrackerCacheInfo();
        cacheInfo.setEntries(this.cacheMap.entrySet().stream().map(entry -> {
            AzureDevopsCompositeCacheKey key = (AzureDevopsCompositeCacheKey)entry.getKey();
            AzureDevopsCachedValues values = (AzureDevopsCachedValues)entry.getValue();
            return new BugTrackerCacheInfo.Entry(Long.valueOf(key.bugTrackerId()), key.path(), values.lastUpdated(), values.lastSuccessfulUpdated(), values.hasCacheError());
        }).toList());
        return cacheInfo;
    }

    AzureDevopsValueCacheQueue getQueue() {
        return this.updateQueue;
    }

    TaskScheduler getTaskScheduler() {
        return this.taskScheduler;
    }

    public StoredCredentialsManager getStoredCredentialsManager() {
        return this.storedCredentialsManager;
    }
}

