/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2018 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
define(["jquery", "squash.translator", "handlebars", "./BaseModel", "./BaseScreen", "./jirasync-tables-resources", 
	"app/util/StringUtil"], 
		function($, translator, Handlebars, BaseModel, BaseScreen, tabres, StringUtils){
	
	
	var dataService = squashtm.app.contextRoot + 'jirasync/exec-plan';

	// load complementary labels (not the table language itself) 
	translator.load({yes : 'label.Yes', no : 'label.No'});
	
	
	/* ****************************************************************
	 * 
	 * 	Table filtering 
	 * 
	  ****************************************************************/
	
	var TableFilter = {
		
		txtFilter : "",
		statuses : [],
		feature : false,
		issue : false,
		covered : false,
		
		reset : function(){
			this.txtFilter = "";
			this.statuses = [];
			this.feature = false;
			this.issue = false;
			this.covered = false;
		},
		
		update : function(){
			this.txtFilter = $("#ticketsel-table-search input").val().toLowerCase();
			this.statuses = $("#ticketsel-table-status input:checked").map(function(){return this.value;}).get();
			
			var synced = $("#ticketsel-table-synccols input:checked").map(function(){return this.value;}).get();
			this.feature = _.contains(synced, 'sync-feat');
			this.issue = _.contains(synced, 'sync-is');
			this.covered = _.contains(synced, 'sync-tc');
		},
		
		// [issue 7215]
		// note : this method is considered unsafe when used on untrusted content
		// in our case it's fine because the server took care of escaping the ticket summary
		unescape : function(encoded){
			return $('<div>').html(encoded).text();
		},
		
		filter : function(ticket){
			
			var retained = true;
			
			if (this.txtFilter.length >0 ){
				var cmp = (ticket.key+' '+ticket.issueType+' '+ this.unescape(ticket.summary) +' '+ ticket.assignee).toLowerCase();
				retained = retained && (cmp.indexOf(this.txtFilter) !== -1);
			}
			
			if (this.statuses.length > 0){
				retained = retained && _.contains(this.statuses, ticket.status);
			}
			
			if (this.filterRole()){
				var retainBecauseFeat = (this.feature && ticket.syncedAsFeature);
				var retainBecauseIssue = (this.issue && ticket.syncedAsIssue);
				var retainBecauseCovered = (this.covered && ticket.featureIsCovered);
				
				retained = retained && (retainBecauseFeat || retainBecauseIssue || retainBecauseCovered);
			}
			
			return retained;
			
		},
		
		filterRole : function(){
			return this.feature || this.issue || this.covered;
		}
	}

	
	
	/*
	 * Register the filtering function 
	 * (see https://datatables.net/examples/plug-ins/range_filtering.html)
	 */
	
	$.fn.dataTable.ext.search.push(function(settings, data, dataIndex){
		
		// early exit if this is not the table we are looking for
		if (settings.sTableId !== 'jirasync-ticketsel-table'){
			return true;
		}
		
		var ticket = settings.aoData[dataIndex]._aData;
		
		return TableFilter.filter(ticket);
		
	});
	
		
	function initTable(table, model){			
		
		var labels = translator.get({yes : 'label.Yes', no : 'label.No'});
		
		table.DataTable({
			
			serverSide : false,
			
			data : model,
			
			bJQueryUI : true,
			dom : 't<"dataTables_footer"lp>',

			scrollCollapse : true,
			paginationType : "squash",
			
			language : tabres.language,
			
			columns : [
				
				// checkbox selection column
				{data: function(ticket, type, value){
					if (type === "display"){
						var selected = (ticket.selected === true) ? 'checked="checked"' : "";
						return '<input class="ticket-select" value="ticket-"'+ticket.key+'" type="checkbox" '+selected+'/>';
					}
					else if (type === "set"){
						ticket.selected = value;
						return;
					}
					else{
						return ticket.selected || false; // for other operations
					}
				}},
				
				// the key
				{data: function(ticket){ return tabres.asBugLink(ticket);}},
				
				// the type
				{data : "issueType"},
				
				// summary
				{data : "summary"},
				
				// status
				{data : "status"},
				
				// assignee
				{data : function(ticket){return (!! ticket.assignee) ? ticket.assignee : "--";}},
				
				// has a sync feature
				{data : function(ticket){return (ticket.syncedAsFeature) ? labels.yes : labels.no;}},

				// feature is covered
				{data : function(ticket){return (ticket.featureIsCovered) ? labels.yes : labels.no;}},
				
				// synced as bugfix
				{data : function(ticket){return (ticket.syncedAsIssue) ? labels.yes : labels.no;}}
				
				
			]
			
		});
		
	}

	
	// ******************** ticket selection ******************
	
	var TicketSelectionModel = BaseModel.extend({
		
		defaults : {
			
			// from megamodel, inited only if strategy is BY_RELEASE
			selectedReleases : [],
			
			// from megamodel,  inited only if strategy is BY_SPRINT
			selectedSprints : [],
			
			// from megamodel, inited only if strategy is BY_JQL,
			jql : { serverId : -1, expression : ""},

			// from megamodel
			strategy : "whatever is inherited from the megamodel",

			// to megamodel 
			selectedTickets : [],
			
			// local data 			
			tickets : []
		},
		
		consumes : ['selectedReleases', 'selectedSprints', 'jql'],
		
		produces : ['selectedTickets'],
		
				
		// *************** data fetch mecanic ******************
		
		ticketPostprocessingMode : 'select-all',
		
		setPostprocessRetainSelected : function(){
			this.ticketPostprocessingMode = 'retain-selected';
		},
		
		couldFetchTickets : function(){
			var strategy = this.strategy();
			switch(strategy){
			case "BY_RELEASE": return this.selectedReleases().length > 0; break;
			case "BY_SPRINT" : return this.selectedSprints().length > 0; break;
			case "BY_JQL" : return ! StringUtils.isBlank(this.jql().expression); break;
			default : throw "not implemented !";
			}
		},

		
		// fetch is special here, because we actually post a query object
		fetchTicketsIfNeeded : function(){
			
			if (this.isFresh()){
				return $.Deferred().resolve().promise();
			}
			
			var self = this;
			
			var payload = this.preparePayload();
			
			return $.ajax({
				url : self.url(),
				type : 'post',
				data : JSON.stringify(payload),
				contentType : 'application/json'
			})
			.done(function(response){
				self.tickets(response);
				if (self.ticketPostprocessingMode === "select-all"){
					self.postprocessSelectAll();
				}
				else{
					self.postprocessRetainSelected();
				}
					
				self.fresh = true; 
				return this;
			});
		},
		
		/*
		 * just select all the tickets
		 */
		postprocessSelectAll : function(){
			var newTickets = this.tickets();
			newTickets.forEach(function(ticket){ ticket.selected = true });
			this.selectedTickets(newTickets);
		},
		
		/*
		 * Compare the newly available tickets and the selected tickets, 
		 * keep selected only those that were already there and don't select the 
		 * others 
		 */
		postprocessRetainSelected : function(){
			var newTickets = this.tickets();
			
			// here we raise the 'selected' flag 
			// for the incoming test cases
			// so that the datatable can use it
			var selectedKeys = _.pluck(this.selectedTickets(), 'key');			
			newTickets.forEach(function(ticket){ ticket.selected = _.contains(selectedKeys, ticket.key); });
			
			// now we detect if any of he previously selected ticket has disappeared from 
			// the new batch of tickets			
			var availableKeys = _.pluck(this.tickets(), 'key');
			var oldSelection = this.selectedTickets();
			var newSelection = oldSelection.filter(function(ticket){ return _.contains(availableKeys, ticket.key) });
			
			// has anything changed ? if so, set the new value to the model
			if (newSelection.length < oldSelection.length){				
				this.selectedTickets(newSelection);
			}
		},
		
		
		url : function(){
			var strategy = this.strategy();
			switch(strategy){
			case "BY_RELEASE" : 
				return dataService + '/jira-projects/releases/issues/search';
				break;
			case "BY_SPRINT" :
				return dataService + '/jira-boards/sprints/issues/search';
				break;
			case "BY_JQL" : 
				return dataService + '/jql-resolver/search';
				break;
			default : throw "not implemented";
			}
			
		},
		
		preparePayload : function(){
			var strategy = this.strategy();
			switch(strategy){
			case "BY_RELEASE" : 
				return this.selectedReleases();
			case "BY_SPRINT" : 
				return this.selectedSprints();
			case "BY_JQL" :
				return this.jql();
			default : throw "not implemented";
			}
		}
		
		
	});
	
	
	// ******************* Ticket selection ****************
	
	var TicketSelectionScreen = BaseScreen.extend({
		
		name : "ticket-select",
		
		el : "#screen-ticket-select",

		template : "#jirsync-plan-ticksel-template",
		
		// ---- rendering 

		_destroyWidgets : function(){
			this.table().DataTable().destroy();
		},
		
		render : function(){
			var self = this;
			this.__renderWait();
			this.model.fetchTicketsIfNeeded()
				.done(function(){
					self.deferredRender();
				})
				.error(function(jqXhr){
					self.__renderAjaxError(jqXhr);
				});
			return this;
		},
		
		templateModel : function(){
			var status = _.chain(this.model.tickets())
				.map(function(ticket){return ticket.status})
				.uniq()
				.value();
			
			return {
				status : status
			};		
		},
		
		deferredRender : function(){
			
			// here the rendering is special because we also need to
			// wait for the table. So the rendering here is a bit 
			// different : the pane is appended to the wait pane
			// so that it is hidden until the table is ready

			this._destroyWidgets();
			var tpl = $(this.template).html();
			var tplFn = Handlebars.compile(tpl);
			var hbarModel = this.templateModel();
			var html = tplFn(hbarModel);
			this.$el.append(html);
			
			var table = this.table();
			
			// this is the actual event that enable 
			// the pane to be shown
			var self = this;
			table.one('init.dt', function(){
				self.$el.find('.jirsync-waitpane').remove();
				self.enableUntilBottom();
			});
			
			// reset the filter
			TableFilter.reset();
			
			// table init
			initTable(table, this.model.tickets());
			
			return this;
			
		},
		
		// ******* status ***********

		isReady : function(){
			var couldFetch = this.model.couldFetchTickets();
			
			return couldFetch;
		},
		
		isComplete : function(){			
			var selection = this.model.selectedTickets();			
			return selection.length > 0;
		},
		
		// ------- data manipulation
		
		events : {
			"click .ticket-select" : "selectTicket",
			"change .ticketsel-filters" : "filter",
			"keyup .ticketsel-filters" : "filter",
			"click .ticketsel-select" : "groupSelect"
		},
		
		selectTicket : function(evt){
			
			var box = $(evt.currentTarget);
			var checked = box.prop('checked'),
				cell = box.parent('td');

			// set the property 'selected' of the datatable model
			// because the datatable needs it for rendering
			this.table().DataTable().cell(cell).data(checked);
			 
			// and now set the model attribute 'selectedTickets'
			this.storeInModel();
			
		},
		
		/*
		 * React to the buttons select all, none, invert.
		 * It must apply only to rows that pass the current filter.
		 */
		groupSelect : function(evt){
			var table = this.table().DataTable();
			var allData = table.data().toArray();
			
			var mode = $(evt.currentTarget).data('select');
			var operation;
			switch(mode){
				case 'all' : operation = function(data){data.selected = true;}; break;
				case 'none' : operation = function(data){data.selected = false;}; break;
				case 'invert' : operation = function(data){data.selected = !data.selected;}; break;
				default : throw "unsupported selection operation : "+mode;
			}
				
			// First, partition the rows that pass the filter and those that don't.
			// Deselect those that don't.
			var filtered = _.partition(allData, function(data){return TableFilter.filter(data);});
			filtered[1].forEach(function(d){d.selected = false});
						
			// Now, for that that pass the filter, apply the operation
			filtered[0].forEach(operation);
			
			// Tell the model to update its data
			this.storeInModel();
			
			// force the redraw of the table
			table.rows().invalidate();			
			
		},
		
		filter : function(){
			var table = this.table().DataTable();
			
			// remember that object at the top of the file ? 
			TableFilter.update();
			
			
			/*
			 * [issue 7203] : the issue was introduced by the fix for #7182. The new business rule that satisfy #7182 and #7203 is now :
			 * 
			 * - the selected/deselected statut of a ticket inside the table is not altered by the filters
			 * - however the effective selection (saved by this#storeInModel) must comply with the filter.
			 * 
			 * Consequently code added for #7182 has been removed. See also comment in #storeInModel.
			 */
			this.storeInModel();
			
			// invalidate and redraw
			table.rows().invalidate();		
			table.draw();

			
		},
		
		storeInModel : function(){
			var selection = this.table().DataTable()
						.rows()
						.data()
						//[issue 7203] must now retain ticket if it is selected and satisfy the filter
						.filter(function(elt){return TableFilter.filter(elt) && elt.selected === true;})
						.toArray();
			
			this.model.selectedTickets(selection);		
			
			// also notify the model it must change its postprocessing behavior
			this.model.setPostprocessRetainSelected();
		},
		
		// ------ DOM getters
		
		// dom getters
		table : function(){
			return this.$el.find('#jirasync-ticketsel-table');
		}
		
	});
	
	
	return {
		model : TicketSelectionModel,
		screen : TicketSelectionScreen
	}
	
	
});