/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2018 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
define(["handlebars", "./BaseModel", "./BaseScreen", "./jirasync-tables-resources"], function(Handlebars, BaseModel, BaseScreen, tabres){
	
	var dataService = squashtm.app.contextRoot + 'jirasync/exec-plan';
	
	
	/* ****************************************************************
	 * 
	 * 	Table filtering
	 * 
	 *  TODO : factor the code with TestCaseSelectionScreen 
	 *  if a third table requires filtering some day
	 * 
	  ****************************************************************/
	
	var TableFilter = {
		
		txtFilter : "",
		
		reset : function(){
			this.txtFilter = "";
		},
		
		update : function(){
			this.txtFilter = $("#testcasesel-table-search input").val().toLowerCase();
		},
		
		filter : function(testcase){
			
			var retained = true;
			
			if (this.txtFilter.length >0 ){
				var cmp = (testcase.name+' '+testcase.reference).toLowerCase();
				retained = retained && (cmp.indexOf(this.txtFilter) !== -1);
			}
			
			return retained;
			
		}
	}

	
	
	/*
	 * Register the filtering function 
	 * (see https://datatables.net/examples/plug-ins/range_filtering.html)
	 */
	$.fn.dataTable.ext.search.push(function(settings, data, dataIndex){
		
		// early exit if this is not the table we are looking for
		if (settings.sTableId !== 'jirasync-tcsel-table'){
			return true;
		}
		
		var ticket = settings.aoData[dataIndex]._aData;
		
		return TableFilter.filter(ticket);
		
	});
	
	
	/* **************************************************************
		table definition
	****************************************************************/

	
	function initTable(table, model){

		
		/* table init
		 * note : this is not a squashtable but a native datatable
		 *
		 * Also note that the datatable model is a shared attribute of the Backbone model
		 *
		 */
		
		table.DataTable({
			serverSide : false,
			
			data : model,
			
			bJQueryUI : true,
			dom : 't<"dataTables_footer"lp>',
			
			language : tabres.language,
			
			scrollCollapse : true,
			paginationType : "squash",
			
			
			columns : [
				// the checkbox columns. Note that this column is settable.
				{data: function(tc, type, value){ 
					if (type === "display"){
						var selected = (tc.selected === true) ? 'checked="checked"' : "";
						return '<input class="tc-select" value="tc-"'+tc.id+'" type="checkbox" '+selected+'/>';
					}
					else if (type === "set"){
						tc.selected = value;
						return;
					}
					else{
						return tc.selected || false; // for other operations
					}
				}},
				
				// tc name column
				{data : function(tc){ return tabres.asTcLink(tc); }},
				
				//requirement column
				{data : function(tc, type){
					if (! tc.hasNewFeatures){ 
						return '--'
					}
					else{
						return tc.requirements.map(tabres.asReqLink).join(',');
					}
				}},
				
				// issues column
				{data : function(tc, type){
					if (! tc.hasBugfixes){
						return "--";
					}
					else{
						return tc.issues.map(tabres.asBugLink).join(','); 
					}
				}}
			]
		});	
		
	}


	
	
	// ******************* test case selection ****************
	
	var TestCaseSelectionModel = BaseModel.extend({
		
		defaults : {
			// in/out attributes from megamodel
			selectedTickets : [],
			selectedTestCases : [],
			
			// fetched
			testCases : [],
			
			// ui state
			statusFilter : []
			
		},
		
		consumes : ['selectedTickets'],
		
		produces : ['selectedTestCases'],
		
		
		//*************** data fetch ********************

		
		url : function(){
			return dataService + '/test-cases/search-by-issues';
		},

		tcPostprocessingMode : 'select-all',
		
		setPostprocessRetainSelected : function(){
			this.tcPostprocessingMode = 'retain-selected';
		},
		
		fetchTestCasesIfNeeded : function(){
			
			if (this.isFresh()){
				return $.Deferred().resolve().promise();
			}
			
			var self = this;
			
			var toSend = this.selectedTickets();
			
			return $.ajax({
				url : self.url(),
				type : 'post',
				data : JSON.stringify(toSend),
				contentType : 'application/json'
			})
			.done(function(response){
				self.testCases(response);
				
				if (self.tcPostprocessingMode === 'select-all'){
					self.postprocessSelectAll();
				}
				else{
					self.postprocessRetainSelected();
				}
					
				self.fresh = true;
				return this;
			});
		},
		
		/*
		 * just select all the test cases
		 */
		postprocessSelectAll : function(){
			var newTcs = this.testCases();
			newTcs.forEach(function(tc){ tc.selected = true });
			this.selectedTestCases(newTcs);
		},
		
		/*
		 * Compare the newly available test cases and the selected testcases, 
		 * keep selected only those that were already there and don't select the 
		 * others 
		 */
		postprocessRetainSelected : function(){
			
			var newTcs = this.testCases();
			
			var selectedIds = _.pluck(this.selectedTestCases(), 'id');			
			newTcs.forEach(function(tc){ tc.selected = _.contains(selectedIds, tc.id); });
			
			var availableIds = _.pluck(this.testCases(), 'id');
			var oldSelection = this.selectedTestCases();
			var newSelection = oldSelection.filter(function(tc){ return _.contains(availableIds, tc.id)});
			
			// has anything changed ? If so, set the new value to the model
			if (newSelection.length < oldSelection.length){
				this.selectedTestCases(newSelection);
			}
		}
		
	});
	
	
	// ******************* Test case selection ****************

	var TestCaseSelectionScreen = BaseScreen.extend({

		name : "testcase-select",
		
		el : "#screen-testcase-select",
		
		template : "#jirsync-plan-tcsel-template",
		
		// *********** rendering *******************
		
		render : function(){
			var self = this;
			this.__renderWait();
			this.model.fetchTestCasesIfNeeded()
				.done(function(){
					self.deferredRender();
				});
			return this;	
		},
		
		_destroyWidgets : function(){
			this.table().DataTable().destroy();
		},
		
		templateModel : function(){
			var status = _.chain(this.model.testCases())
				// collect all the requirements and issues
				.map(function(tc){return tc.requirements.concat(tc.issues);})
				.flatten()
				// collect the statuses
				.map(function(thing){return thing.status})
				.uniq()
				.value();
			
			return {
				status : status
			};
			
		},
	
		deferredRender : function(){
			
			// here the rendering is special because we also need to
			// wait for the table. So the rendering here is a bit 
			// different : the pane is appended to the wait pane
			// so that it is hidden until the table is ready
			
			this._destroyWidgets();
			var tpl = $(this.template).html();
			var tplFn = Handlebars.compile(tpl);
			var html = tplFn(this.templateModel());
			this.$el.append(html);
			
			var table = this.table();
			
			// this is the actual event that enable 
			// the pane to be shown
			var self = this;
			table.one('init.dt', function(){
				self.$el.find('.jirsync-waitpane').remove();
				self.enableUntilBottom();
			});
			
			// reset the table filter
			TableFilter.reset();
			
			// table init
			initTable(table, this.model.testCases());
			
			return this;
		},
		
		// ********** status *********************
		
		isReady : function(){
			var selection = this.model.selectedTickets();			
			return selection.length > 0;
		},
		
		isComplete : function(){
			var selection = this.model.selectedTestCases();			
			return selection.length > 0;
		},
		
		
		// ***** event handling ******************
		
		events : {
			"click .tc-select" 				: "selectTestCase",
			"keyup .testcasesel-filters"	: "filter",
			"click .testcasesel-select" 	: "groupSelect"
		},
		
		
		selectTestCase : function(evt){
			var box = $(evt.currentTarget);
			var checked = box.prop('checked'),
				cell = box.parent('td');
			
			// set the property 'selected' of the datatable model
			// because the datatable needs it for rendering
			this.table().DataTable().cell(cell).data(checked);
			 
			// and now set the model attribute 'selectedTestCases'
			this.storeInModel();
		},
		
		/*
		 * React to the buttons select all, none, invert.
		 * It must apply only to rows that pass the current filter.
		 */
		groupSelect : function(evt){
			var table = this.table().DataTable();
			var allData = table.data().toArray();
			
			var mode = $(evt.currentTarget).data('select');
			var operation;
			switch(mode){
				case 'all' : operation = function(data){data.selected = true;}; break;
				case 'none' : operation = function(data){data.selected = false;}; break;
				case 'invert' : operation = function(data){data.selected = !data.selected;}; break;
				default : throw "unsupported selection operation : "+mode;
			}
				
			// First, partition the rows that pass the filter and those that don't.
			// Deselect those that don't.
			var filtered = _.partition(allData, function(data){return TableFilter.filter(data);});
			filtered[1].forEach(function(d){d.selected = false});
						
			// Now, for that that pass the filter, apply the operation
			filtered[0].forEach(operation);
			
			// Tell the model to update its data
			this.storeInModel();
			
			// force the redraw of the table
			table.rows().invalidate();			
			
		},
		
		filter : function(){
			var table = this.table().DataTable();
			
			// remember that object at the top of the file ? 
			TableFilter.update();
			
			/*
			 * [issue 7203] : the issue was introduced by the fix for #7182. The new business rule that satisfy #7182 and #7203 is now :
			 * 
			 * - the selected/deselected statut of a test case inside the table is not altered by the filters
			 * - however the effective selection (saved by this#storeInModel) must comply with the filter.
			 * 
			 * Consequently code added for #7182 has been removed. See also comment in #storeInModel.
			 */
			this.storeInModel();
			
			// invalidate and redraw
			table.rows().invalidate();		
			table.draw();
		},
		
		storeInModel : function(){
			var selection = this.table().DataTable()
								.rows()
								.data()
								//[issue 7203] must now retain ticket if it is selected and satisfy the filter
								.filter(function(elt){return TableFilter.filter(elt) && elt.selected === true;})
								.toArray();
			
			this.model.selectedTestCases(selection);
						
			// also notify the model it must change its postprocessing behavior
			this.model.setPostprocessRetainSelected();
		},
		
		// ************ helpers ****************
		
		table : function(){
			return $('#jirasync-tcsel-table');
		},
		
		visibleRows : function(){
			var table = this.table();
			var $rows = table.find('tbody tr');
			return table.DataTable().rows($rows);
		}
		
	});	
	
	
	
	return {
		model : TestCaseSelectionModel,
		screen : TestCaseSelectionScreen
	}
	
	
});