/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2018 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
/*
 * Optional screens if the chosen strategy is BY_RELEASE.
 * 
 * Note that here we got two screens that share the same model (instead of one per model).
 * 
 */
define(["jquery",  "./BaseModel", "./BaseScreen", "./classes", "underscore", 
	"squash.configmanager", "squash.dateutils", "squash.translator", "app/util/StringUtil",
	"jqueryui", "jeditable", "jeditable.datepicker"], 
		function( $, BaseModel, BaseScreen, classes, _, confman, dateutils, translator, StringUtils){
	
	
	var dataService = squashtm.app.contextRoot + 'jirasync/exec-plan';
	
	// **************** strategy BY_RELEASE model ********************
	
	/*
	 * This model is common to both the ProjectSelectionScreen 
	 * and ReleaseSelectionScreen.
	 */
	
	var ByReleaseModel = BaseModel.extend({
		
		consumes : ['strategy'],
		produces : ['selectedReleases'],
		
		defaults : {
			
			// --- the ultimate goal of this model ----
			
			// this is the final result of the couple of screens 'project select' and 'release select'
			// (as indicated by the 'produces' array)
			selectedReleases : [],
			
			// --- local UI state data ----
			scope : "SQ_PROJ",
			projectId : null,
			projectName : null,
			
			// manually selected JIRA projects
			userJiraSelection : [],
			
			// filtering options
			optReleaseDateEnabled : false,
			optReleaseDate : null,
			optMostRecentEnabled : false,
			optMostRecent : 1,
			optNameLikeEnabled : false,
			optNameLike : "",
			
			// ---- data fetched from the server -----
			
			// jira projects for the current squash project
			forSelectedProject : [],
			
			// all available JIRA projects
			forAllProjects : [],
			
			// the set of available releases that match the selected projects and filters.
			// On screen 'Project selection', the number of releases is displayed in the release counter.
			// On screen 'Release selection', the content is displayed in a list.
			releases : []
			
		},		
		
		init : function(mega){
			
			// regarding invalidation, this model is special because it does not 
			// listen to the megamodel but to itself.
			this.on({
				'change:scope' : this.refetchReleases,
				'change:userJiraSelection': this.refetchReleases,
				'change:optReleaseDateEnabled' : this.refetchReleases,
				'change:optReleaseDate' : this.refetchReleases,
				'change:optNameLikeEnabled' : this.refetchReleases,
				'change:optNameLike' : this.refetchReleases,
				'change:optMostRecentEnabled' : this.refetchReleases,
				'change:optMostRecent' : this.refetchReleases
			});
		
			// initialize the required attributes
			return {
				projectId : mega.selectedNode().projectId,
				projectName : mega.selectedNode().projectName
			};
			
		},
		
		// ***************** flags **********************
		
		/*
		 * This flag helps ensuring that the projects are fetched only once
		 */
		flagFetchProjects : true,
		
		/*
		 * The release fetch postprocessing logic is driven by this flag.
		 * A value of "select-all" will call the 'postprocessSelectAll' behavior,
		 * while value "retain-selected" will call the 'postprocessRetainSelected' behavior
		 * 
		 * (see 'fetchReleasesIfNeeded')
		 */		
		releasePostprocessMode : "select-all",
		
		// this changes the postprocessing behavior
		setPostprocessRetainSelected : function(){
			this.releasePostprocessMode = "retain-selected";
		},
	
		
		// ****** internal event handling ***************
		
		refetchReleases : function(){
			/*
			 *  we need to 
			 *  1/ invalidate the attribute 'releases' (this tells the next view to redraw)
			 *  2/ fetch the releases immediately because we need to count them.
			 */ 
			this.invalidate();
			this.fetchReleasesIfNeeded();
		},

		
		// ***** project selection part ******
				
		// if needs to fetch projects, return the promise on the ajax requests
		// if it's already loaded, return a resolved promise anyway
		fetchProjectsIfNeeded : function(){
			
			if (! this.flagFetchProjects){
				return $.Deferred().resolve().promise();
			}
			
			var self = this;
			var url = dataService + '/jira-projects?selected-squash-project-id='+this.attributes.projectId;
			
			return $.ajax({
				url : url,
				type : 'GET',
				dataType : 'json'
			})
			.done(function(json){
				self.forSelectedProject(json.forSelectedProject);
				self.forAllProjects(json.forAllProjects);
				self.flagFetchProjects = false;
				return this;
			});
			
		},
		
		// ***** release selection part ********
		
		getEffectiveProjectSelection : function(){
			var scope = this.scope();
			return (scope==="SQ_PROJ") ? this.forSelectedProject() : this.userJiraSelection();
		},

		
		// if needs to fetch releases, return the promise on the ajax requests
		// if it's already loaded, return a resolved promise anyway
		fetchReleasesIfNeeded : function(){
			
			if (this.isFresh()){
				return $.Deferred().resolve().promise();
			}
			
			var self = this;
			var url = dataService + '/jira-projects/releases/search';
			
			var payload = this.prepareFetchReleasesPayload();
			
			return $.ajax({
				url : url,
				type : 'POST',
				dataType : 'json',
				contentType : 'application/json',
				data : JSON.stringify(payload)
			})
			.done(function(json){
				self.releases(json.result);	
				
				// postprocessing
				if (self.releasePostprocessMode === "select-all"){
					self.postprocessSelectAll();
				}
				else{
					self.postprocessRetainSelected();
				}
				
				self.fresh = true;
				
				// force the triggering of 'change:releases' otherwise the ProjectSelection 
				// screen will have a problem
				self.trigger('change:releases');
				
				return this;
			});
		},
		
		prepareFetchReleasesPayload : function(){
			var projects = this.getEffectiveProjectSelection();
			
			var relSince = (this.optReleaseDateEnabled()) ? this.optReleaseDate() : null;
			var nameLike = (this.optNameLikeEnabled()) ? this.optNameLike() : null;
			var mostRecent = (this.optMostRecentEnabled()) ? this.optMostRecent() : null;
			
			return {
				releasedSince : relSince,
				nameLike : nameLike,
				mostRecentReleasesLimit : mostRecent,
				jiraProjects : projects
			};
		},
		
		
		/*
		 * This postprocessing behavior will simply consider 
		 * that all newly fetched sprints are selected 
		 */
		postprocessSelectAll : function(){
			this.selectedReleases(this.releases());
		},
		
		/*
		 * This postprocessing behavior will compare the existing 
		 * selection with the new set of available releases
		 * and retain selected only those that are still present.
		 */
		postprocessRetainSelected : function(){
			// turning the ProjectReleases to a string form 
			// will greatly simplify the impact calculus
			var stockAsString = classes.ProjectReleases.toStrings(this.releases());
			var selecAsString = classes.ProjectReleases.toStrings(this.selectedReleases());
			
			// using the string form, remove any selected item that is not available anymore
			// in the new stock of item
			var retainedAsString = _.filter(selecAsString, function(selected){
				return _.contains(stockAsString, selected);
			});
			
			// has anything been removed in the process ?
			// if so, set the new value. The event triggered will signal downstream 
			// models/screens that they need updates too
			if (retainedAsString.length < selecAsString.length){
				var parsed = classes.ProjectReleases.fromStrings(retainedAsString);
				this.selectedReleases(parsed);
			}	
		}
		
	});
	
	
	
	
	// **** By Release Strategy : Project selection **********
	
	var ProjectSelectionScreen = BaseScreen.extend({

		name : "project-select",
		
		el : "#screen-project-select",
			
		template :"#jirsync-plan-byrel-projsel-template",
		
		userDformat : null,		
		storageDformat : dateutils.ISO_8601,
		
		initialize : function(){
			BaseScreen.prototype.initialize.apply(this, arguments);
			this.userDformat = translator.get('squashtm.dateformatShort.datepicker');
		},
		
		// ********** status *********** 
		
		isComplete : function(){
			return this.model.getEffectiveProjectSelection().length > 0;
		},
		
		onModelInvalidated : function(){
			// This method is invoked when the model is currently refetching 
			// the available releases. This signals that the counter should enter the wait mode
			this.setReleaseCounterWaitMode();
		},
		
		// ******** rendering **********
		
		_destroyWidgets : function(){
			this.releaseDatePicker().editable('destroy');
		},
		
		render : function(){

			var self = this;
			
			this.__renderWait();
			this.model.fetchProjectsIfNeeded()
					.done(function(){
						BaseScreen.prototype.__renderValid.apply(self, arguments);
						self.initReleaseDatePicker();
						// fetching the releases now will update the counter
						self.model.fetchReleasesIfNeeded()
							.done(function(){
								self.setReleaseCounterDisplayMode()
							});
					});

			return this;
		},
		
		initReleaseDatePicker : function(){		
			
			var datepicker = this.releaseDatePicker();
			var pickerconf = confman.getStdDatepicker();
			var jedconf = confman.getStdJeditable();
			
			var self = this;
			
			datepicker.editable(
				function(value){					
					var date = datepicker.find('input').datepicker('getDate');
					var asString = dateutils.format(date, this.storageDformat);
					datepicker.trigger('change', {value : asString});
					return value;
				}, {
				type : "datepicker",
				datepicker : pickerconf
			});
			
			datepicker.editable("disable");
		},
		
		
		templateModel : function(){
			
			var scope = this.model.scope();
			var selected = this.model.userJiraSelection();
			
			var reldateEnabled = this.model.optReleaseDateEnabled();
			var reldate = this.model.optReleaseDate();
			
			var mostRecentEnabled = this.model.optMostRecentEnabled();
			var mostRecent = this.model.optMostRecent();

			var nameLikeEnabled = this.model.optNameLikeEnabled();
			var nameLike = this.model.optNameLike();
			
			var subpaneEnabled = (scope === "ALL_PROJ");
			
			return {
				mainpane : {
					radiocheck : scope,
					
					sqProjName : this.model.projectName(),
					forThisSquashProject : this.model.forSelectedProject().join(','),
				},
				
				subpane : {
					enabled : subpaneEnabled,
					projects : this.model.forAllProjects().map(function(proj){
						return {
							name : proj,
							checked : _.contains(selected, proj),
							disabled : !subpaneEnabled
						}
					})
				},
				
				opts : {
					releaseDate : {
						checked : reldateEnabled,
						date : reldate,
						disabled : ! reldateEnabled 
					},
					mostRecent : {
						checked : mostRecentEnabled,
						limit : mostRecent,
						disabled : ! mostRecentEnabled
					},
					nameLike : {
						checked : nameLikeEnabled,
						name : nameLike,
						disabled : ! nameLikeEnabled
					}
				}
				
			};
		},

		
		// *********** state mutators *******************

		events : {
			'change input[name="byrel-scope"]' : 'changeScope',
			'change .jirsync-subpane input' : 'changeSelection',
				
			'change .jirsync-release-date-picker' : "changeReleaseDate",
			'change .jirsync-release-date-check' : 'enableReleaseDate',

			'change .jirsync-mostrecent-check' : 'enableMostRecent',
			'change .jirsync-mostrecent-input' : 'changeMostRecent',

			'change .jirsync-namelike-check' : 'enableNameLike',
			'change .jirsync-namelike-input' : 'changeNameLike',
		},
		
		modelEvents : {
			'change:releases' : "setReleaseCounterDisplayMode"
		},

		changeScope : function(evt){
			var value = $(evt.currentTarget).val();
			this.model.scope(value);
			
			this.enableSubpane();
		},
		
		changeSelection : function(){
			var projects = this.subpane()
								.find('input:checked')
								.map(function(){
									return this.value;
								})
								.get();
			this.model.userJiraSelection(projects);
		},
		
		enableReleaseDate : function(evt){
			var dp = this.releaseDatePicker();
			var enabled = $(evt.currentTarget).prop('checked');
			this.model.optReleaseDateEnabled(enabled);
			
			var method = (enabled) ? "enable" : "disable";
			
			// enable/disable
			dp.editable(method);
			
			// also set style
			if (enabled) 
				dp.removeClass('jirsync-disabled'); 
			else 
				dp.addClass('jirsync-disabled');
			
		},
		
		changeReleaseDate : function(evt, value){
			this.model.optReleaseDate(value.value);
		},
		
		enableMostRecent : function(evt, value){
			var enabled = $(evt.currentTarget).prop('checked');
			this.model.optMostRecentEnabled(enabled);
			
			this.$el.find('.jirsync-mostrecent-input').prop('disabled', !enabled);
		},
		
		changeMostRecent : function(evt, value){
			var input = $(evt.currentTarget);
			var value = input.val();
			if (StringUtils.isBlank(value) || isNaN(value)){
				input.next().show();
			}
			else{
				this.model.optMostRecent(parseInt(value,10));
				input.next().hide();
			}
		},
		
		enableNameLike : function(evt, value){
			var enabled = $(evt.currentTarget).prop('checked');
			this.model.optNameLikeEnabled(enabled);
			
			this.$el.find('.jirsync-namelike-input').prop('disabled', !enabled);
		},
		
		changeNameLike : function(evt, value){
			var value = $(evt.currentTarget).val();
			this.model.optNameLike(value);
		},
		
		// ******** view elements **************************
		
		releaseDateCheckbox : function(){
			return this.$el.find('.jirsync-release-date-check');
		},
		
		releaseDatePicker : function(){
			return this.$el.find('.jirsync-release-date-picker');
		},
		
		subpane : function(){
			return this.$el.find('.jirsync-subpane');
		},
		
		enableSubpane : function(){
			var scope = this.model.scope();
			var enabled = (scope === "ALL_PROJ");
			
			var opacity = (enabled) ? 1.0 : 0.3;
			var chkDisabled = !enabled;
			
			var subpane = this.$el.find('.jirsync-subpane');
			subpane.css('opacity', opacity);
			subpane.find('input').prop('disabled', chkDisabled);
		},
		
		setReleaseCounterWaitMode : function(){
			this.$el.find('.source-counter-wait').show();
			this.$el.find('.source-counter-message').hide();
		},
		
		setReleaseCounterDisplayMode : function(){
			this.$el.find('.source-counter-wait').hide();
			this.$el.find('.source-counter-message').show();
			
			var cnt = _.chain(this.model.releases())
						.map(function(projrel){ return projrel.releases})
						.flatten()
						.value()
						.length;
			this.$el.find('.source-counter').text(cnt);
		}
		
	});
	
	
	// **** By Release Strategy : Release selection **********
	
	var ReleaseSelectionScreen = BaseScreen.extend({
	
		name : "release-select",
		
		el : "#screen-release-select",
	
		template :"#jirsync-plan-byrel-relsel-template",
				
		render : function(){
			var self = this;
			this.__renderWait();
			this.model.fetchReleasesIfNeeded()
					.done(function(){
						BaseScreen.prototype.__renderValid.apply(self, arguments);
					});
			return this;
		},
		
		templateModel : function(){
			
			var selecAsString = classes.ProjectReleases.toStrings(this.model.selectedReleases());
			
			var releaseModel = this.model.releases().map(function(projrel){
				var project = projrel.project;
				var releases = projrel.releases.map(function(release){
					var asString = project+"-"+release.id;
					var checkmix = {
						checked : _.contains(selecAsString, asString)
					};
					return _.extendOwn(checkmix, release);
					
				});
				return new classes.ProjectReleases(project, releases);
			});
			
			return {
				releases : releaseModel
			}			
		},
		
		// ********* status change ***********
		
		isReady : function(){
			var selection = this.model.getEffectiveProjectSelection()
			
			return selection.length > 0 ;
		},

		isComplete : function(){
			var selection = this.model.selectedReleases();
			
			return selection.length > 0;
					
		},
		
		// ********** events *****************
		
		events : {
			'change input' : 'changeReleases',
			'click .relsel-select' : 'groupSelect'
		},
		
		changeReleases : function(){
			
			var checkboxes = this.$el.find('input:checked');
			
			// turn to a suitable representation
			var asStrings = checkboxes.map(function(){
				var relId = this.value;
				var proj = $(this).parents('ul:first').data('project');
				return proj+'-'+relId;
			}).get();
			
			// now parse it with the helper functions
			var releases = classes.ProjectReleases.fromStrings(asStrings);
			
			this.model.selectedReleases(releases);
			
			// also, now the user has interacted with the model,
			// we need to instruct it to change its release fetch postprocessing behavior
			this.model.setPostprocessRetainSelected();
		},
		
		groupSelect : function(evt){
			// define what to do with the checkboxes
			var mode = $(evt.currentTarget).data('select');
			var operation;
			switch(mode){
			case 'all' : operation = function(){this.checked=true}; break;
			case 'none' : operation = function(){this.checked=false;}; break;
			case 'invert' : operation = function(){this.checked = !this.checked;}; break;
			default : throw "unsupported selection operation : "+mode;
			}
			
			// apply the operation
			this.$el.find('input[type="checkbox"]').each(operation);
			
			// now save the new selection
			this.changeReleases();
		}
		
	});
	
	
	return {
		model : ByReleaseModel,
		projselScreen : ProjectSelectionScreen,
		relselScreen : ReleaseSelectionScreen
	};

	
});