/*
 *     This file is part of the Jira Agile synchronization connector for Squash TM (plugin.requirement.xsquash4jira) project.
 *     Copyright (C) 2017 - 2018 Henix, henix.fr - All Rights Reserved
 *
 *     Unauthorized copying of this file, via any medium is strictly prohibited
 *     Proprietary and confidential
 *
 * 	 (C)Henix. Tous droits réservés.
 *
 * 	Avertissement : ce programme est protégé par la loi relative au droit d'auteur et par les conventions internationales. Toute reproduction ou distribution partielle ou totale du logiciel, par quelque moyen que ce soit, est strictement interdite.
 */
/* 
 * 
 * BaseModel
 * 
 * 
 * Base class for our view models. See JirasyncExecplanMainView for an introduction and the 
 * big picture.
 * 
 * 
 * -------------- features -----------------
 * 
 * 1/ Fancy Accessors
 * ------------------
 * 
 * Defining the default values will autogenerate shorthands for getter/setters
 * 
 * this allows to access model attributes as follow 
 * 
 * - model.attribute() instead of model.get('attribute') and
 * - model.attribute(value) instead of model.set('attribute', value)
 * 
 * 
 * 
 * 2/ Composite models 
 * --------------------
 * 
 * Some model will be given the MegaModel (see MegaModel) as an option : 
 * 
 * new MyScreenModel({attrs}, {mega : themegamodel})
 *
 * If so, the model will automatically inherit and expose all the attributes of the mega
 * model, but any get/set operation will be forwarded to the megamodel instead of keeping them
 * for its own. This mechanism allow to share data between models via a centralized place 
 * (that's what the megamodel is for).
 * 
 * For instance, for a screen model that delegates to the megamodel, 
 * a call to set('selectedTestCases', [testcases]) will be automatically forwarded 
 * to the megamodel instead of keeping it to itself, and likewise get('selectedTestCases')
 * will automatically fetch them from megamodel too.
 * 
 *  This is compatible with the fancy accessors, as such the example above 
 *  can also be written as calling selectedTestCases([testcases]) (for set) and
 *  selectedTestCases() (for get). To this end, these properties still need to 
 *  be declared in the defaults hash as described in 1/ Fancy Accessors, however 
 *  their default value will be ignored (we don't want to erase the megamodel 
 *  with dummy values).
 *  
 *  3/ Init function
 *  ------------------
 *  
 *  The method 'init' allows a model to initialize itself. The parameter is
 *  the MegaModel if available. This method is called every time the corresponding 
 *  view is shown. It is useful to initialize properties that are specific to this 
 *  model and view.
 *  
 *  4/ Model freshness and invalidation
 *  -----------------------------------
 *  
 *  The ultimate goal of a screen model is to read/write data from/to the MegaModel :   
 *  they require data to work with (the screen cannot be rendered
 *  without them) and produce new data (when the user interacts with their screen).
 *  
 *  A model is said to be fresh if its current content has not been made 
 *  obsolete retroactively when the user changed something in a step 
 *  that affects its prerequisites.
 *  
 *  Because the output of a model is the input of the next model, and because when 
 *  the input change the model will invalidate the output that was obtained 
 *  from it, any change in a property of the MegaModel will induce a cascade of 
 *  invalidation events. The models that come downstream (according to the ScreenFlow)
 *  are then likely to lose their 'fresh' status. 
 *  
 *  A Model is responsible to watch for changes in the megamodel that would invalidate it, 
 *  and notify its Screen when this is the case. The screen usually responds by changing 
 *  its status.
 * 
 */ 
define(["backbone", "underscore"], function(Backbone, _){


	
	function accessor(ppt, value){
		if (value !== undefined){
			var toSet = {};
			toSet[ppt] = value;
			this.set(toSet);
		}
		else{
			return this.get(ppt);
		}
	}
	
	var BaseModel = Backbone.Model.extend({
		
		// the syncable attributes, see documentation above
		dataAttributes : [],
		
		_delegated : [],
		
		/*
		 * Attributes of the megamodel that this model 
		 * needs to work with. If 
		 * one of these change the model 
		 * will be invalidated. 
		 * 
		 * If that array is left empty, the model
		 * is assumed to be always valid.
		 */
		consumes : [],
		
		/*
		 * Attributes of the megamodel that this model
		 * will produce.
		 */
		produces : [],
		
		constructor : function(attributes, options){
			
			// if the megamodel is present, initialize the _delegated array
			// and the megamodel events
			if (!! options && !!options.mega){
				this.mega = options.mega;
				this._delegated = _.keys(options.mega.defaults);
				this._watchInvalidation(options.mega);
			}
			
			var defaults = this.defaults;	
			
			
			/*
			 * This will add the said properties for each instance of a view when created.
			 * A more accurate approach would be to override Backbone.Model.extend and 
			 * make this part of the extension mechanism, but I don't want to mess 
			 * with Backbone too much.
			 * 
			 * Note that here the default attribute will be that one of the subclasses.
			 */
			if (!! defaults){
				for (var prop in defaults){					
					this[prop] = _.partial(accessor, prop);
				}
			}
			
			
			/*
			 * Now that our fancy accessors are created, we need to remove the delegated 
			 * properties from the defaults. If we don't do it, the default values will 
			 * be applied to the megamodel and we don't want that.
			 */
			
			var _delegated = this._delegated;
			for (var prop in defaults){
				if (_.contains(_delegated, prop)){
					delete defaults[prop];
				}
			}
			
			Backbone.Model.apply(this, arguments);

		},
		
		initialize : function(attributes, options){
			var inited = {};
			
			if (!! this.mega){
				// call the init function				
				var inited = this.init(this.mega);				
				$.extend(this.attributes, inited, attributes);
			}
			
			Backbone.Model.prototype.initialize.call(this, attributes, options);
		},
		
		// opportunity to scrap properties from the mega model
		init : function(mega){
			return {};
		},
		
		
		// returns the model attributes 
		// only for the given array of attributes
		collect : function(selectedAttributes){
			var collected = {};
			for (var i=0; i < selectedAttributes.length; i++){
				var prop = selectedAttributes[i];
				collected[prop] = this.get(prop);
			}
			return _.clone(collected);
		},
		
		// get and set delegate to the megamodel
		set : function(hash, opts){
			var own = _.omit(hash, this._delegated);
			var toMega = _.pick(hash, this._delegated);
			Backbone.Model.prototype.set.call(this, own, opts);
			if (! _.isEmpty(toMega)){
				this.mega.set(toMega, opts);
			}
		},
		
		get : function(attr){
			if (_.contains(this._delegated, attr)){
				return this.mega.get(attr);
			}
			else{
				return Backbone.Model.prototype.get.call(this, attr);
			}
		},
		
		
		// *********** Freshness management ***************
		
		/*
		 * The main flag regarding the model freshness. 
		 * Remember to set it to true everytime the model
		 * has updated its data (when GETing clean json 
		 * objects for instance).
		 */
		fresh : false,
	
		/*
		 * A Screen will ask the model if its valid 
		 * via this method.
		 */
		isFresh : function(){
			return this.fresh;
		},
		
		
		_watchInvalidation : function(mega){
			if (this.consumes.length === 0){
				this.fresh = true;
			}
			else{
				var boundInvalidate = _.bind(this.invalidate, this);
				
				// create the event hash
				var evts = {};
				this.consumes.forEach(function(attribute){
					evts['change:'+attribute] = boundInvalidate;
				});
				
				// now bind the megamodel to it
				mega.on(evts);
			}
		},
		
		/*
		 * Implementors may manually call this method when 
		 * they wish to invalidate the model for whatever reason 
		 */
		invalidate : function(){
			this.fresh = false;
			var mega = this.mega;
			this.produces.forEach(function(attribute){
				mega.trigger('change:'+attribute);
			})
			
			this.trigger('model-invalidated');
		}
		

	});

	
	return BaseModel;
	
	
});