/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.service.concurrent;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.squashtest.tm.core.foundation.logger.Logger;
import org.squashtest.tm.core.foundation.logger.LoggerFactory;
import org.squashtest.tm.service.concurrent.EntityLockRef;
import org.squashtest.tm.service.concurrent.EntityLockTimeoutException;
import org.squashtest.tm.service.concurrent.EntityLockedSection;

public final class EntityLockManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(EntityLockManager.class);
    private static int LOCK_TIMEOUT_SECONDS = 10;
    private static final int LOCK_CLEANUP_INTERVAL_SECONDS = 300;
    private static final Map<EntityLockRef, ManagedEntityLock> locks = new ConcurrentHashMap<EntityLockRef, ManagedEntityLock>();
    private static final ScheduledExecutorService cleanupExecutor = Executors.newScheduledThreadPool(1, r -> {
        Thread t = new Thread(r, "EntityLockManager-Cleanup");
        t.setDaemon(true);
        return t;
    });

    static {
        EntityLockManager.setupCleanupThread();
    }

    private EntityLockManager() {
    }

    public static Object processWithLock(EntityLockRef entityLockRef, EntityLockedSection section) throws Throwable {
        return EntityLockManager.processWithLock(List.of(entityLockRef), section);
    }

    public static Object processWithLock(Collection<EntityLockRef> entityLockRefs, EntityLockedSection section) throws Throwable {
        Map<EntityLockRef, ManagedEntityLock> managedLocksToProcess = EntityLockManager.getManagedLocks(entityLockRefs);
        ArrayList<ManagedEntityLock> acquiredManagedEntityLocks = new ArrayList<ManagedEntityLock>();
        try {
            EntityLockManager.acquireLocks(managedLocksToProcess, acquiredManagedEntityLocks);
            Object object = section.perform();
            return object;
        }
        finally {
            if (TransactionSynchronizationManager.isActualTransactionActive()) {
                EntityLockManager.releaseLockOnTransactionEnding(acquiredManagedEntityLocks);
            } else {
                EntityLockManager.releaseLocks(acquiredManagedEntityLocks);
            }
        }
    }

    private static void releaseLockOnTransactionEnding(final List<ManagedEntityLock> locks) {
        TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new TransactionSynchronization(){

            public void afterCompletion(int status) {
                EntityLockManager.releaseLocks(locks);
            }
        });
    }

    private static Map<EntityLockRef, ManagedEntityLock> getManagedLocks(Collection<EntityLockRef> entityLockRefs) {
        LinkedHashMap<EntityLockRef, ManagedEntityLock> locksMap = new LinkedHashMap<EntityLockRef, ManagedEntityLock>();
        ArrayList<EntityLockRef> sortedEntityLockRefs = new ArrayList<EntityLockRef>(entityLockRefs);
        sortedEntityLockRefs.sort(EntityLockRef::compareTo);
        for (EntityLockRef ref : sortedEntityLockRefs) {
            ManagedEntityLock managedLock = EntityLockManager.acquireReference(ref);
            locksMap.put(ref, managedLock);
        }
        return locksMap;
    }

    private static ManagedEntityLock acquireReference(EntityLockRef ref) {
        while (true) {
            ManagedEntityLock lock;
            if ((lock = locks.computeIfAbsent(ref, k -> new ManagedEntityLock())).acquireReference()) {
                if (locks.get(ref) == lock) {
                    return lock;
                }
                lock.releaseReference();
                continue;
            }
            locks.remove(ref, lock);
        }
    }

    private static void acquireLocks(Map<EntityLockRef, ManagedEntityLock> managedLocksToProcess, List<ManagedEntityLock> acquiredManagedEntityLocks) throws InterruptedException {
        for (Map.Entry<EntityLockRef, ManagedEntityLock> entry : managedLocksToProcess.entrySet()) {
            EntityLockRef ref = entry.getKey();
            ManagedEntityLock managedEntityLock = entry.getValue();
            try {
                if (!managedEntityLock.lock.tryLock(LOCK_TIMEOUT_SECONDS, TimeUnit.SECONDS)) {
                    throw new EntityLockTimeoutException("Could not acquire lock for entity %s within timeout of %s seconds".formatted(ref, LOCK_TIMEOUT_SECONDS));
                }
                acquiredManagedEntityLocks.add(managedEntityLock);
                LOGGER.trace("Acquired lock for entity {}", new Object[]{ref});
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw e;
            }
        }
        LOGGER.debug("Acquired {} locks", new Object[]{acquiredManagedEntityLocks.size()});
    }

    private static void releaseLocks(List<ManagedEntityLock> acquiredManagedEntityLocks) {
        int i = acquiredManagedEntityLocks.size() - 1;
        while (i >= 0) {
            ManagedEntityLock managedEntityLock = acquiredManagedEntityLocks.get(i);
            try {
                if (managedEntityLock.lock.isHeldByCurrentThread()) {
                    managedEntityLock.lock.unlock();
                    LOGGER.trace("Released lock for entity", new Object[0]);
                }
            }
            catch (Exception e) {
                LOGGER.warn("Error unlocking entity lock", (Throwable)e);
            }
            --i;
        }
        for (ManagedEntityLock managedEntityLock : acquiredManagedEntityLocks) {
            managedEntityLock.releaseReference();
        }
        LOGGER.debug("Released {} locks and references", new Object[]{acquiredManagedEntityLocks.size()});
    }

    private static void setupCleanupThread() {
        cleanupExecutor.scheduleWithFixedDelay(EntityLockManager::cleanupUnusedLocks, 300L, 300L, TimeUnit.SECONDS);
        Runtime.getRuntime().addShutdownHook(new Thread(EntityLockManager::shutdownExecutor));
    }

    private static void cleanupUnusedLocks() {
        LOGGER.trace("Starting cleanup of unused locks. Current lock map size: {}", new Object[]{locks.size()});
        ArrayList<Map.Entry<EntityLockRef, ManagedEntityLock>> entries = new ArrayList<Map.Entry<EntityLockRef, ManagedEntityLock>>(locks.entrySet());
        for (Map.Entry entry : entries) {
            EntityLockManager.attemptCleanup((EntityLockRef)entry.getKey(), (ManagedEntityLock)entry.getValue());
        }
        LOGGER.trace("Finished cleanup of unused locks. Current lock map size: {}", new Object[]{locks.size()});
    }

    private static void attemptCleanup(EntityLockRef ref, ManagedEntityLock managedEntityLock) {
        if (!managedEntityLock.markForDeletion()) {
            LOGGER.trace("Lock for entity {} is in use, cannot mark for deletion", new Object[]{ref});
            return;
        }
        if (managedEntityLock.lock.tryLock()) {
            try {
                if (managedEntityLock.isSafeToDelete() && locks.remove(ref, managedEntityLock)) {
                    LOGGER.debug("Cleaned up unused lock for entity {}", new Object[]{ref});
                }
                managedEntityLock.unmarkForDeletion();
                LOGGER.trace("Lock for entity {} became active during cleanup, unmarking", new Object[]{ref});
            }
            finally {
                managedEntityLock.lock.unlock();
            }
        } else {
            managedEntityLock.unmarkForDeletion();
            LOGGER.trace("Lock for entity {} is busy, cannot cleanup", new Object[]{ref});
        }
    }

    private static void shutdownExecutor() {
        cleanupExecutor.shutdown();
        LOGGER.info("Shutting down EntityLockManager cleanup executor...", new Object[0]);
        try {
            if (!cleanupExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                LOGGER.warn("EntityLockManager cleanup executor did not terminate in time, forcing shutdown.", new Object[0]);
                cleanupExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            LOGGER.warn("Interrupted while waiting for EntityLockManager cleanup executor to terminate", (Throwable)e);
            cleanupExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    static void setLockTimeoutSeconds(int lockTimeoutSeconds) {
        LOCK_TIMEOUT_SECONDS = lockTimeoutSeconds;
    }

    private static class ManagedEntityLock {
        private static final int MARKED_FOR_DELETION = Integer.MIN_VALUE;
        private final AtomicInteger referenceCount = new AtomicInteger(0);
        final ReentrantLock lock = new ReentrantLock(true);

        private ManagedEntityLock() {
        }

        boolean acquireReference() {
            int current;
            do {
                if ((current = this.referenceCount.get()) != Integer.MIN_VALUE) continue;
                return false;
            } while (!this.referenceCount.compareAndSet(current, current + 1));
            return true;
        }

        void releaseReference() {
            int current;
            do {
                if ((current = this.referenceCount.get()) == Integer.MIN_VALUE) {
                    return;
                }
                if (current > 0) continue;
                throw new IllegalStateException("Attempted to release reference when count is " + current);
            } while (!this.referenceCount.compareAndSet(current, current - 1));
        }

        boolean markForDeletion() {
            return this.referenceCount.compareAndSet(0, Integer.MIN_VALUE);
        }

        void unmarkForDeletion() {
            this.referenceCount.compareAndSet(Integer.MIN_VALUE, 0);
        }

        boolean isSafeToDelete() {
            if (!this.lock.isHeldByCurrentThread()) {
                throw new IllegalStateException("isSafeToDelete must be called while holding the lock");
            }
            return this.referenceCount.get() == Integer.MIN_VALUE && this.lock.getHoldCount() == 1 && this.lock.getQueueLength() == 0;
        }
    }
}

