/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.service.internal.servers;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.GeneralSecurityException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.transaction.Transactional;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Service;
import org.squashtest.tm.core.foundation.logger.Logger;
import org.squashtest.tm.core.foundation.logger.LoggerFactory;
import org.squashtest.tm.domain.project.GenericProject;
import org.squashtest.tm.domain.servers.AuthenticationProtocol;
import org.squashtest.tm.domain.servers.BasicAuthenticationCredentials;
import org.squashtest.tm.domain.servers.Credentials;
import org.squashtest.tm.domain.servers.StoredCredentials;
import org.squashtest.tm.domain.users.User;
import org.squashtest.tm.security.UserContextHolder;
import org.squashtest.tm.service.feature.FeatureManager;
import org.squashtest.tm.service.internal.repository.UserDao;
import org.squashtest.tm.service.internal.servers.Crypto;
import org.squashtest.tm.service.internal.servers.ManageableBasicAuthCredentials;
import org.squashtest.tm.service.internal.servers.StoredContentHandlers;
import org.squashtest.tm.service.security.PermissionEvaluationService;
import org.squashtest.tm.service.servers.EncryptionKeyChangedException;
import org.squashtest.tm.service.servers.ManageableCredentials;
import org.squashtest.tm.service.servers.MissingEncryptionKeyException;
import org.squashtest.tm.service.servers.ServerAuthConfiguration;
import org.squashtest.tm.service.servers.StoredCredentialsManager;

@Transactional
@Service
public class StoredCredentialsManagerImpl
implements StoredCredentialsManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(StoredCredentialsManagerImpl.class);
    private static final String OR_CURRENT_USER_OWNS_CREDENTIALS = " or authentication.name == #username";
    private static final String DELETE_ALL_USER_CREDENTIALS = "StoredCredentials.deleteAllUserCredentialsByUserId";
    private static final String DELETE_ALL_STORED_CREDENTIALS_FOR_SERVERS = "StoredCredentials.deleteAllStoredCredentialsForServers";
    private static final String DELETE_ALL_PROJECT_CREDENTIALS = "StoredCredentials.deleteAllProjectCredentialsByProjectId";
    private static final String JACKSON_TYPE_ID_ATTR = "@class";
    private static final String REFUSED_TO_STORE_CREDENTIALS_OF_TYPE = "Refused to store credentials of type '";
    private static final String BUSINESS_RULES_FORBID = "' : business rules forbid ";
    @PersistenceContext
    private EntityManager em;
    @Inject
    private FeatureManager features;
    @Inject
    private UserDao userDao;
    @Inject
    private PermissionEvaluationService permissionService;
    @Value(value="${squash.crypto.secret}")
    private char[] secret = new char[0];
    private ObjectMapper objectMapper;
    private boolean caseInsensitive = false;

    @Override
    public boolean isSecretConfigured() {
        if (this.secret.length == 0 || this.secret[0] == '\u0000') {
            return false;
        }
        int i = 0;
        while (i < this.secret.length) {
            char digit = this.secret[i];
            if (digit != ' ' && digit != '\t') {
                return true;
            }
            ++i;
        }
        return false;
    }

    @Override
    public char[] getSecret() {
        return this.secret;
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN') or authentication.name == #username")
    public void storeUserCredentials(long serverId, String username, ManageableCredentials credentials) {
        this.checkUserLevelStorageAllowed(credentials);
        this.storeContent(this.buildUserCredentialsHandler(serverId, username), credentials);
    }

    @Override
    public void storeCurrentUserCredentials(long serverId, ManageableCredentials credentials) {
        this.checkUserLevelStorageAllowed(credentials);
        String username = UserContextHolder.getUsername();
        this.storeContent(this.buildUserCredentialsHandler(serverId, username), credentials);
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN') or authentication.name == #username")
    public ManageableCredentials findUserCredentials(long serverId, String username) {
        return this.unsecuredFindUserCredentials(serverId, username);
    }

    @Override
    public ManageableCredentials findCurrentUserCredentials(long serverId) {
        String username = UserContextHolder.getUsername();
        return this.unsecuredFindUserCredentials(serverId, username);
    }

    @Override
    public ManageableCredentials unsecuredFindUserCredentials(long serverId, String username) {
        return this.unsecuredFindContent(this.buildUserCredentialsHandler(serverId, username));
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN') or authentication.name == #username")
    public void deleteUserCredentials(long serverId, String username) {
        this.deleteContent(this.buildUserCredentialsHandler(serverId, username));
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public void deleteAllUserCredentials(long userId) {
        this.deleteAllUserCredentialsById(userId);
    }

    @Override
    public void storeAppLevelCredentials(long serverId, ManageableCredentials credentials) {
        this.permissionService.checkAtLeastOneProjectManagementPermissionOrAdmin();
        this.checkAppLevelStorageAllowed(credentials);
        this.storeContent(this.buildAppLevelHandler(serverId), credentials);
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public ManageableCredentials findAppLevelCredentials(long serverId) {
        return this.unsecuredFindAppLevelCredentials(serverId);
    }

    @Override
    public ManageableCredentials unsecuredFindAppLevelCredentials(long serverId) {
        return this.unsecuredFindContent(this.buildAppLevelHandler(serverId));
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public void deleteAppLevelCredentials(long serverId) {
        this.deleteContent(this.buildAppLevelHandler(serverId));
    }

    @Override
    public void deleteAllServerCredentials(List<Long> serverIds) {
        if (serverIds.isEmpty()) {
            return;
        }
        LOGGER.info("Deleting all credentials for server #{}", new Object[]{serverIds});
        Query query = this.em.createNamedQuery(DELETE_ALL_STORED_CREDENTIALS_FOR_SERVERS);
        query.setParameter("serverIds", serverIds);
        query.executeUpdate();
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public void storeServerAuthConfiguration(long serverId, ServerAuthConfiguration conf) {
        this.storeContent(this.buildAuthConfigurationHandler(serverId), conf);
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public ServerAuthConfiguration findServerAuthConfiguration(long serverId) {
        return this.unsecuredFindServerAuthConfiguration(serverId);
    }

    @Override
    public ServerAuthConfiguration unsecuredFindServerAuthConfiguration(long serverId) {
        return this.unsecuredFindContent(this.buildAuthConfigurationHandler(serverId));
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public void deleteServerAuthConfiguration(long serverId) {
        this.deleteContent(this.buildAuthConfigurationHandler(serverId));
    }

    @Override
    @PreAuthorize(value="hasPermission(#projectId, 'org.squashtest.tm.domain.project.Project', 'MANAGE_PROJECT')  or hasRole('ROLE_ADMIN')")
    public void storeProjectCredentials(long serverId, long projectId, ManageableCredentials credentials) {
        this.checkProjectLevelStorageAllowed(credentials);
        this.storeContent(this.buildProjectHandler(serverId, projectId), credentials);
    }

    @Override
    public ManageableCredentials findProjectCredentials(long serverId, long projectId) {
        return this.unsecuredFindContent(this.buildProjectHandler(serverId, projectId));
    }

    @Override
    @PreAuthorize(value="hasPermission(#projectId, 'org.squashtest.tm.domain.project.Project', 'MANAGE_PROJECT')  or hasRole('ROLE_ADMIN')")
    public void deleteProjectCredentials(long serverId, long projectId) {
        this.deleteContent(this.buildProjectHandler(serverId, projectId));
    }

    @Override
    @PreAuthorize(value="hasPermission(#projectId, 'org.squashtest.tm.domain.project.Project', 'MANAGE_PROJECT')  or hasRole('ROLE_ADMIN')")
    public void deleteAllProjectCredentials(long projectId) {
        LOGGER.info("Deleting all project credentials for project #{}", new Object[]{projectId});
        Query query = this.em.createNamedQuery(DELETE_ALL_PROJECT_CREDENTIALS);
        query.setParameter("projectId", (Object)projectId);
        query.executeUpdate();
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public void storeReportingCacheCredentials(long serverId, ManageableCredentials credentials) {
        this.storeContent(this.buildReportingCacheHandler(serverId), credentials);
    }

    @Override
    public ManageableCredentials findReportingCacheCredentials(long serverId) {
        return this.unsecuredFindContent(this.buildReportingCacheHandler(serverId));
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public void deleteReportingCacheCredentials(long serverId) {
        this.deleteContent(this.buildReportingCacheHandler(serverId));
    }

    private void checkAppLevelStorageAllowed(ManageableCredentials credentials) {
        if (!credentials.allowsAppLevelStorage()) {
            throw new IllegalArgumentException(REFUSED_TO_STORE_CREDENTIALS_OF_TYPE + credentials.getImplementedProtocol() + BUSINESS_RULES_FORBID + "to store such credentials as application-level credentials");
        }
    }

    private void checkUserLevelStorageAllowed(ManageableCredentials credentials) {
        if (!credentials.allowsUserLevelStorage()) {
            throw new IllegalArgumentException(REFUSED_TO_STORE_CREDENTIALS_OF_TYPE + credentials.getImplementedProtocol() + BUSINESS_RULES_FORBID + "to store such credentials for human users");
        }
    }

    private void checkProjectLevelStorageAllowed(ManageableCredentials credentials) {
        if (!credentials.allowsProjectLevelStorage()) {
            throw new IllegalArgumentException(REFUSED_TO_STORE_CREDENTIALS_OF_TYPE + credentials.getImplementedProtocol() + BUSINESS_RULES_FORBID + "to store such credentials as project-level credentials");
        }
    }

    StoredContentHandlers.UserCredentialsHandler buildUserCredentialsHandler(long serverId, String username) {
        return new StoredContentHandlers.UserCredentialsHandler(serverId, username, () -> this.loadUserOrNull(username));
    }

    StoredContentHandlers.AuthConfigurationHandler buildAuthConfigurationHandler(long serverId) {
        return new StoredContentHandlers.AuthConfigurationHandler(serverId);
    }

    StoredContentHandlers.AppLevelHandler buildAppLevelHandler(long serverId) {
        return new StoredContentHandlers.AppLevelHandler(serverId);
    }

    StoredContentHandlers.ProjectHandler buildProjectHandler(long serverId, long projectId) {
        return new StoredContentHandlers.ProjectHandler(serverId, projectId, () -> (GenericProject)this.em.find(GenericProject.class, (Object)projectId));
    }

    StoredContentHandlers.ReportingCacheHandler buildReportingCacheHandler(long serverId) {
        return new StoredContentHandlers.ReportingCacheHandler(serverId);
    }

    private <TYPE> void storeContent(StoredContentHandlers.StoredContentHandler<TYPE> handler, Object content) {
        if (!this.isSecretConfigured()) {
            throw new MissingEncryptionKeyException();
        }
        Crypto.EncryptionOutcome outcome = null;
        try {
            outcome = this.toEncryptedForm(content);
        }
        catch (IOException | GeneralSecurityException ex) {
            LOGGER.error("Could encrypt the content because the JRE doesn't support the specified encryption algorithms.", new Object[0]);
            throw new RuntimeException(ex);
        }
        StoredCredentials sc = null;
        try {
            sc = this.loadStoredContent(handler);
            sc.setEncryptedCredentials(outcome.getEncryptedText());
        }
        catch (NoResultException noResultException) {
            sc = handler.buildNewStoredCredentials(this.em, outcome);
            this.em.persist((Object)sc);
        }
    }

    private <TYPE> TYPE unsecuredFindContent(StoredContentHandlers.StoredContentHandler<TYPE> handler) {
        if (!this.isSecretConfigured()) {
            throw new MissingEncryptionKeyException();
        }
        LOGGER.debug("loading stored content for content handler '{}'", new Object[]{handler});
        Crypto crypto = new Crypto(Arrays.copyOf(this.secret, this.secret.length));
        StoredCredentials sc = null;
        String strDecrypt = null;
        try {
            Class<TYPE> deserializationClass = handler.getDeserializationClass();
            sc = this.loadStoredContent(handler);
            strDecrypt = crypto.decrypt(sc.getEncryptedCredentials());
            Object object = this.objectMapper.readValue(strDecrypt, deserializationClass);
            return (TYPE)object;
        }
        catch (NoResultException noResultException) {
            LOGGER.debug("Content not found.", new Object[0]);
            return null;
        }
        catch (UnsupportedEncodingException | GeneralSecurityException cryptoException) {
            LOGGER.error("Decryption failed probably because the encryption key changed. Less likely, is also can be that JRE doesn't support the specified encryption algorithms.", new Object[0]);
            throw new EncryptionKeyChangedException(cryptoException);
        }
        catch (Exception ex) {
            if (strDecrypt == null) {
                LOGGER.debug("The decryption failed for unknown reasons.", new Object[0]);
                throw new RuntimeException(ex);
            }
            TYPE TYPE = this.fallbackOrDie(sc, strDecrypt);
            return TYPE;
        }
        finally {
            crypto.dispose();
        }
    }

    private <TYPE> TYPE fallbackOrDie(StoredCredentials sc, String strDecrypt) {
        LOGGER.debug("The data format is wrong. Perhaps an instance of BasicAuthenticationCredentials, which cannot now be stored directly, needs migration ?", new Object[0]);
        try {
            return (TYPE)this.migrateToNewFormat(strDecrypt, sc);
        }
        catch (IOException | ClassCastException definitelyWrong) {
            LOGGER.error("something has gone definitely wrong.", new Object[0]);
            LOGGER.error(definitelyWrong.getMessage(), (Throwable)definitelyWrong);
            throw this.investigateDeserializationError(strDecrypt, definitelyWrong);
        }
        catch (GeneralSecurityException cryptoException) {
            LOGGER.error("encryption exception while trying to migrate and restore old credentials ! Was the secret key changed ?", new Object[0]);
            throw new EncryptionKeyChangedException(cryptoException);
        }
    }

    private <TYPE> void deleteContent(StoredContentHandlers.StoredContentHandler<TYPE> handler) {
        try {
            StoredCredentials sc = this.loadStoredContent(handler);
            this.em.remove((Object)sc);
        }
        catch (NoResultException noResultException) {}
    }

    private Crypto.EncryptionOutcome toEncryptedForm(Object credentials) throws IOException, GeneralSecurityException {
        Crypto crypto = new Crypto(Arrays.copyOf(this.secret, this.secret.length));
        try {
            String strCreds = null;
            try {
                strCreds = this.objectMapper.writeValueAsString(credentials);
            }
            catch (JsonProcessingException ex) {
                LOGGER.error("an error occured while storing the credentials due to serialization error ", (Throwable)ex);
                throw new RuntimeException(ex);
            }
            Crypto.EncryptionOutcome encryptionOutcome = crypto.encrypt(strCreds);
            return encryptionOutcome;
        }
        finally {
            crypto.dispose();
        }
    }

    private StoredCredentials loadStoredContent(StoredContentHandlers.StoredContentHandler<?> handler) {
        Query query = handler.getLocateQuery(this.em);
        return (StoredCredentials)query.getSingleResult();
    }

    private User loadUserOrNull(String username) {
        User user = null;
        if (username != null) {
            user = this.caseInsensitive ? this.userDao.findUserByCiLogin(username) : this.userDao.findUserByLogin(username);
        }
        return user;
    }

    private void deleteAllUserCredentialsById(long userId) {
        LOGGER.info("Deleting all user credentials for user #{}", new Object[]{userId});
        Query query = this.em.createNamedQuery(DELETE_ALL_USER_CREDENTIALS);
        query.setParameter("userId", (Object)userId);
        query.executeUpdate();
    }

    private RuntimeException investigateDeserializationError(String failedDeser, Throwable cause) {
        try {
            Map asMap = (Map)this.objectMapper.readValue(failedDeser, Map.class);
            String clazz = (String)asMap.get(JACKSON_TYPE_ID_ATTR);
            return new RuntimeException("missing implementation for ManageableCredentials type '" + clazz + "', or that type does not implement '" + ManageableCredentials.class.getName() + "'", cause);
        }
        catch (IOException e) {
            return new EncryptionKeyChangedException(e);
        }
    }

    @PostConstruct
    void initialize() {
        ObjectMapper om = new ObjectMapper();
        om.addMixIn(Credentials.class, InternalCredentialsMixin.class);
        om.addMixIn(ManageableCredentials.class, InternalManageableCredentialsMixin.class);
        om.addMixIn(ServerAuthConfiguration.class, InternalServerAuthConfigurationMixin.class);
        this.objectMapper = om;
        this.caseInsensitive = this.features.isEnabled(FeatureManager.Feature.CASE_INSENSITIVE_LOGIN);
    }

    private ManageableCredentials migrateToNewFormat(String strDecrypt, StoredCredentials sc) throws IOException, GeneralSecurityException {
        LOGGER.debug("attempting migration of the deprecated stored credentials", new Object[0]);
        ManageableCredentials fixed = this.tryAsBasicAuth(strDecrypt);
        Crypto.EncryptionOutcome outcome = this.toEncryptedForm(fixed);
        sc.setEncryptedCredentials(outcome.getEncryptedText());
        LOGGER.debug("migration completed", new Object[0]);
        return fixed;
    }

    private ManageableCredentials tryAsBasicAuth(String serialized) throws IOException {
        BasicAuthenticationCredentials creds = (BasicAuthenticationCredentials)this.objectMapper.readValue(serialized, BasicAuthenticationCredentials.class);
        return new ManageableBasicAuthCredentials(creds.getUsername(), creds.getPassword());
    }

    @JsonTypeInfo(include=JsonTypeInfo.As.PROPERTY, use=JsonTypeInfo.Id.CLASS)
    @JsonInclude
    static interface InternalCredentialsMixin {
        @JsonIgnore
        public AuthenticationProtocol getImplementedProtocol();
    }

    @JsonTypeInfo(include=JsonTypeInfo.As.PROPERTY, use=JsonTypeInfo.Id.CLASS)
    @JsonInclude
    static interface InternalManageableCredentialsMixin {
        @JsonIgnore
        public AuthenticationProtocol getImplementedProtocol();
    }

    @JsonTypeInfo(include=JsonTypeInfo.As.PROPERTY, use=JsonTypeInfo.Id.CLASS)
    @JsonInclude
    static interface InternalServerAuthConfigurationMixin {
        @JsonIgnore
        public AuthenticationProtocol getImplementedProtocol();
    }
}

