/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.service.internal.user;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Base64;
import java.util.Date;
import java.util.Map;
import java.util.UUID;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.squashtest.tm.domain.users.ApiToken;
import org.squashtest.tm.domain.users.ApiTokenPermission;
import org.squashtest.tm.domain.users.User;
import org.squashtest.tm.exception.JwtTokenImpossibleExtractionException;
import org.squashtest.tm.exception.apitoken.WrongExpiryDateException;
import org.squashtest.tm.exception.user.ApiTokenCreationException;
import org.squashtest.tm.exception.user.ApiTokenDeletionException;
import org.squashtest.tm.exception.user.ApiTokenFetchingException;
import org.squashtest.tm.exception.user.InfrastructureAdminException;
import org.squashtest.tm.security.UserContextHolder;
import org.squashtest.tm.service.internal.dto.ApiTokenDto;
import org.squashtest.tm.service.internal.repository.ApiTokenDao;
import org.squashtest.tm.service.internal.repository.CustomApiTokenDao;
import org.squashtest.tm.service.jwt.JwtTokenService;
import org.squashtest.tm.service.user.ApiTokenService;
import org.squashtest.tm.service.user.UserAccountService;
import org.squashtest.tm.service.user.UserManagerService;

@Service
@Transactional
public class ApiTokenServiceImpl
implements ApiTokenService {
    private static final String INFRASTRUCTURE_ADMIN_TOKEN_NAME = "squash-cloud-admin-token";
    private final UserAccountService userAccountService;
    private final ApiTokenDao apiTokenDao;
    private final CustomApiTokenDao customApiTokenDao;
    private final JwtTokenService jwtTokenService;
    private final UserManagerService userManagerService;
    @Value(value="${squash.rest-api.jwt.secret:#{null}}")
    private String jwtSecret;
    @PersistenceContext
    EntityManager entityManager;

    public ApiTokenServiceImpl(UserAccountService userAccountService, ApiTokenDao apiTokenDao, CustomApiTokenDao customApiTokenDao, JwtTokenService jwtTokenService, UserManagerService userManagerService) {
        this.userAccountService = userAccountService;
        this.apiTokenDao = apiTokenDao;
        this.customApiTokenDao = customApiTokenDao;
        this.jwtTokenService = jwtTokenService;
        this.userManagerService = userManagerService;
    }

    @Override
    public ApiTokenDto generateApiToken(String name, Date expiryDate, String permissions) {
        ApiTokenServiceImpl.checkExpiryDateConformity(expiryDate);
        User user = this.userAccountService.findCurrentUser();
        return this.doGenerateApiToken(name, expiryDate, permissions, user);
    }

    @Override
    public ApiTokenDto generateApiTokenForInfrastructureAdmin(User user) {
        if (!user.isInfrastructureAdmin()) {
            throw new InfrastructureAdminException("The user is not an infrastructure admin.");
        }
        if (this.jwtSecret == null) {
            throw new InfrastructureAdminException("Cannot generate API token because the `squash.rest-api.jwt.secret` property is not set. Please configure this property to enable token generation.");
        }
        return this.doGenerateApiToken(INFRASTRUCTURE_ADMIN_TOKEN_NAME, ApiTokenServiceImpl.generateExpiryDateForInfrastructureAdminToken(), ApiTokenPermission.READ_WRITE.name(), user);
    }

    @Override
    public Page<ApiToken> findAllByUserId(long userId, Pageable pageable) {
        User currentUser = this.userManagerService.findByLogin(UserContextHolder.getUsername());
        if (currentUser.getId() == userId) {
            return this.apiTokenDao.findAllByUserId(userId, pageable);
        }
        throw new ApiTokenFetchingException("Unable to retrieve tokens belonging to another user.");
    }

    @Override
    public void deletePersonalApiToken(long tokenId) {
        ApiToken apiToken = (ApiToken)this.apiTokenDao.findById(tokenId).orElseThrow();
        if (!UserContextHolder.getUsername().equals(apiToken.getUser().getLogin())) {
            throw new ApiTokenDeletionException("Cannot delete a token that does not belong to the current user.");
        }
        this.apiTokenDao.deleteById(tokenId);
    }

    @Override
    public void selfDestroyApiToken(String token) {
        String tokenUuid = this.extractUuidFromJwtToken(token);
        this.apiTokenDao.deleteByUuid(tokenUuid);
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public ApiTokenDto generateApiTokenForTechnicalAccount(long userId, String name, Date expiryDate, String permissions) {
        User user = (User)this.entityManager.find(User.class, (Object)userId);
        if (!this.customApiTokenDao.isTechnicalAccount(userId)) {
            throw new ApiTokenCreationException();
        }
        ApiTokenServiceImpl.checkExpiryDateConformity(expiryDate);
        return this.doGenerateApiToken(name, expiryDate, permissions, user);
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public Page<ApiToken> findAllTechnicalAccountApiTokens(long userId, Pageable pageable) {
        if (this.customApiTokenDao.isTechnicalAccount(userId)) {
            return this.apiTokenDao.findAllByUserId(userId, pageable);
        }
        throw new ApiTokenFetchingException("Administrator can only retrieve tokens from a technical account.");
    }

    @Override
    @PreAuthorize(value="hasRole('ROLE_ADMIN')")
    public void deleteTechnicalAccountApiToken(long tokenId) {
        ApiToken apiToken = (ApiToken)this.apiTokenDao.findById(tokenId).orElseThrow();
        if (!this.customApiTokenDao.isTechnicalAccount(apiToken.getUser().getId())) {
            throw new ApiTokenDeletionException("Administrator can only delete tokens from a technical account.");
        }
        this.apiTokenDao.deleteById(tokenId);
    }

    private static void checkExpiryDateConformity(Date expiryDate) {
        if (ApiTokenServiceImpl.isDateAfterOneYearFromToday(expiryDate)) {
            throw new WrongExpiryDateException("Expiry date cannot be set later one year from today.");
        }
        if (ApiTokenServiceImpl.isDateBeforeTomorrow(expiryDate)) {
            throw new WrongExpiryDateException("Expiry date cannot be set before tomorrow.");
        }
    }

    private ApiTokenDto doGenerateApiToken(String name, Date expiryDate, String permissions, User user) {
        UUID uuid = UUID.randomUUID();
        ApiToken newApiToken = this.persistApiToken(user, uuid.toString(), name, expiryDate, permissions);
        Date jwtExpiryDate = ApiTokenServiceImpl.parseDateAtMidnightInUtc(newApiToken);
        String generatedJwtToken = this.jwtTokenService.generateJwt(newApiToken.getUser().getId().toString(), newApiToken.getUuid(), jwtExpiryDate, newApiToken.getCreatedOn(), ApiTokenPermission.valueOf((String)permissions).name());
        return new ApiTokenDto(newApiToken, generatedJwtToken);
    }

    private static boolean isDateAfterOneYearFromToday(Date givenDate) {
        LocalDate localGivenDate = givenDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        LocalDate today = LocalDate.now();
        LocalDate oneYearFromToday = today.plusYears(1L);
        return localGivenDate.isAfter(oneYearFromToday);
    }

    private static boolean isDateBeforeTomorrow(Date givenDate) {
        LocalDate localGivenDate = givenDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        LocalDate tomorrow = LocalDate.now().plusDays(1L);
        return localGivenDate.isBefore(tomorrow);
    }

    private static Date parseDateAtMidnightInUtc(ApiToken newApiToken) {
        LocalDate localDate = LocalDate.parse(newApiToken.getExpiryDate());
        LocalDateTime localDateTime = LocalDateTime.of(localDate, LocalTime.MIDNIGHT);
        Instant instant = localDateTime.toInstant(ZoneOffset.UTC);
        return Date.from(instant);
    }

    private static Date generateExpiryDateForInfrastructureAdminToken() {
        LocalDate today = LocalDate.now();
        LocalDate oneYearFromToday = today.plusYears(1L);
        return Date.from(oneYearFromToday.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    private String extractUuidFromJwtToken(String token) {
        Map claims;
        String[] parts = token.split("\\.");
        if (parts.length < 2) {
            throw new IllegalArgumentException("Invalid JWT token");
        }
        String payload = new String(Base64.getUrlDecoder().decode(parts[1]));
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            claims = (Map)objectMapper.readValue(payload, (TypeReference)new TypeReference<Map<String, Object>>(){});
        }
        catch (JsonProcessingException e) {
            throw new JwtTokenImpossibleExtractionException((Exception)((Object)e));
        }
        return claims.get("uuid").toString();
    }

    private ApiToken persistApiToken(User user, String uuid, String name, Date expiryDate, String permissions) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        String expiryDateString = simpleDateFormat.format(expiryDate);
        ApiToken apiToken = new ApiToken(uuid, user, name, new Date(), UserContextHolder.getUsername(), expiryDateString, ApiTokenPermission.valueOf((String)permissions).name());
        return (ApiToken)this.apiTokenDao.save(apiToken);
    }
}

