/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.service.internal.execution;

import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.transaction.Transactional;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Service;
import org.squashtest.tm.domain.campaign.SprintStatus;
import org.squashtest.tm.domain.execution.Execution;
import org.squashtest.tm.domain.execution.ExecutionStatus;
import org.squashtest.tm.domain.execution.ExploratoryExecution;
import org.squashtest.tm.domain.execution.ExploratoryExecutionEvent;
import org.squashtest.tm.domain.execution.ExploratoryExecutionEventType;
import org.squashtest.tm.domain.execution.ExploratoryExecutionRunningState;
import org.squashtest.tm.domain.execution.LatestExploratoryExecutionEvent;
import org.squashtest.tm.domain.users.User;
import org.squashtest.tm.exception.campaign.SprintClosedException;
import org.squashtest.tm.exception.exploratoryexecution.IllegalActionForPausedExploratoryExecutionException;
import org.squashtest.tm.exception.exploratoryexecution.IllegalActionForStoppedExploratoryExecutionException;
import org.squashtest.tm.exception.exploratoryexecution.IllegalExploratoryExecutionEventSequenceException;
import org.squashtest.tm.exception.exploratoryexecution.IllegalExploratoryExecutionStateTransitionException;
import org.squashtest.tm.security.UserContextHolder;
import org.squashtest.tm.service.annotation.CheckBlockingMilestone;
import org.squashtest.tm.service.annotation.Id;
import org.squashtest.tm.service.execution.ExploratoryExecutionService;
import org.squashtest.tm.service.internal.repository.ExploratoryExecutionDao;
import org.squashtest.tm.service.internal.repository.ExploratoryExecutionEventDao;
import org.squashtest.tm.service.internal.repository.UserDao;
import org.squashtest.tm.service.internal.repository.display.SprintDisplayDao;
import org.squashtest.tm.service.security.UserContextService;

@Transactional
@Service
public class ExploratoryExecutionServiceImpl
implements ExploratoryExecutionService {
    private final ExploratoryExecutionEventDao exploratoryExecutionEventDao;
    private final UserDao userDao;
    private final UserContextService userContextService;
    private final SprintDisplayDao sprintDisplayDao;
    private final ExploratoryExecutionDao exploratoryExecutionDao;
    @PersistenceContext
    private EntityManager entityManager;

    public ExploratoryExecutionServiceImpl(ExploratoryExecutionEventDao exploratoryExecutionEventDao, UserDao userDao, UserContextService userContextService, SprintDisplayDao sprintDisplayDao, ExploratoryExecutionDao exploratoryExecutionDao) {
        this.exploratoryExecutionEventDao = exploratoryExecutionEventDao;
        this.userDao = userDao;
        this.userContextService = userContextService;
        this.sprintDisplayDao = sprintDisplayDao;
        this.exploratoryExecutionDao = exploratoryExecutionDao;
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'EXECUTE')  or hasRole('ROLE_ADMIN')")
    @CheckBlockingMilestone(entityType=Execution.class)
    public void startOrResumeExploratoryExecution(@Id long executionId, String author) {
        this.checkSprintStatusAllowModification(executionId);
        ExploratoryExecutionRunningState currentState = this.findExploratoryExecutionRunningState(executionId);
        if (ExploratoryExecutionRunningState.NEVER_STARTED.equals((Object)currentState)) {
            this.startSession(executionId, author, currentState);
        } else {
            this.resumeSession(executionId, author, currentState);
        }
    }

    private void startSession(long executionId, String author, ExploratoryExecutionRunningState currentState) {
        if (!ExploratoryExecutionRunningState.NEVER_STARTED.equals((Object)currentState)) {
            throw new IllegalExploratoryExecutionStateTransitionException.AlreadyRunning(ExploratoryExecutionEventType.START);
        }
        this.createEvent(executionId, author, ExploratoryExecutionEventType.START);
        this.updateExecutionMetadata(executionId);
    }

    private void resumeSession(long executionId, String author, ExploratoryExecutionRunningState currentState) {
        if (ExploratoryExecutionRunningState.NEVER_STARTED.equals((Object)currentState) || ExploratoryExecutionRunningState.RUNNING.equals((Object)currentState)) {
            throw new IllegalExploratoryExecutionStateTransitionException.AlreadyRunning(ExploratoryExecutionEventType.RESUME);
        }
        this.createEvent(executionId, author, ExploratoryExecutionEventType.RESUME);
        this.updateExecutionMetadata(executionId);
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'EXECUTE')  or hasRole('ROLE_ADMIN')")
    public void pauseExploratoryExecution(@Id long executionId, String author) {
        this.checkSprintStatusAllowModification(executionId);
        ExploratoryExecutionRunningState currentState = this.findExploratoryExecutionRunningState(executionId);
        if (!ExploratoryExecutionRunningState.RUNNING.equals((Object)currentState)) {
            throw new IllegalExploratoryExecutionStateTransitionException.AlreadyPaused();
        }
        this.createEvent(executionId, author, ExploratoryExecutionEventType.PAUSE);
        this.updateExecutionMetadata(executionId);
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'EXECUTE')  or hasRole('ROLE_ADMIN')")
    public void stopExploratoryExecution(@Id long executionId, String author) {
        this.checkSprintStatusAllowModification(executionId);
        ExploratoryExecutionRunningState currentState = this.findExploratoryExecutionRunningState(executionId);
        if (ExploratoryExecutionRunningState.NEVER_STARTED.equals((Object)currentState) || ExploratoryExecutionRunningState.STOPPED.equals((Object)currentState)) {
            throw new IllegalExploratoryExecutionStateTransitionException.AlreadyStopped();
        }
        this.createEvent(executionId, author, ExploratoryExecutionEventType.STOP);
        this.updateExecutionMetadata(executionId);
    }

    private void createEvent(long executionId, String author, ExploratoryExecutionEventType eventType) {
        ExploratoryExecution exploratoryExecution = (ExploratoryExecution)this.entityManager.find(ExploratoryExecution.class, (Object)executionId);
        this.exploratoryExecutionEventDao.save(new ExploratoryExecutionEvent(exploratoryExecution, new Date(), author, eventType));
    }

    @Override
    public void updateExecutionMetadata(long executionId) {
        Date lastExecutedOn = new Date();
        String lastExecutedBy = this.userContextService.getUsername();
        ExploratoryExecution exploratoryExecution = (ExploratoryExecution)this.entityManager.find(ExploratoryExecution.class, (Object)executionId);
        exploratoryExecution.setLastExecutedOn(lastExecutedOn);
        exploratoryExecution.setLastExecutedBy(lastExecutedBy);
        exploratoryExecution.setExecutionStatus(ExecutionStatus.RUNNING);
        exploratoryExecution.updateTestPlanItemLastExecution(this.userDao.findUserByLogin(lastExecutedBy));
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'READ') or hasRole('ROLE_ADMIN')")
    public ExploratoryExecutionRunningState findExploratoryExecutionRunningState(@Id long executionId) {
        List<ExploratoryExecutionEvent> events = this.exploratoryExecutionEventDao.findByExploratoryExecution_IdOrderByDateDesc(executionId);
        if (events.isEmpty()) {
            return ExploratoryExecutionRunningState.NEVER_STARTED;
        }
        return switch (events.get(0).getEventType()) {
            case ExploratoryExecutionEventType.STOP -> ExploratoryExecutionRunningState.STOPPED;
            case ExploratoryExecutionEventType.PAUSE -> ExploratoryExecutionRunningState.PAUSED;
            case ExploratoryExecutionEventType.START, ExploratoryExecutionEventType.RESUME -> ExploratoryExecutionRunningState.RUNNING;
            default -> throw new MatchException(null, null);
        };
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'READ') or hasRole('ROLE_ADMIN')")
    public LatestExploratoryExecutionEvent fetchLatestExploratoryExecutionEvent(@Id Long executionId) {
        List<ExploratoryExecutionEvent> events = this.exploratoryExecutionEventDao.findByExploratoryExecution_IdOrderByDate(executionId);
        if (events.isEmpty()) {
            return null;
        }
        Date latestEventTimestamp = events.get(events.size() - 1).getDate();
        return new LatestExploratoryExecutionEvent(latestEventTimestamp, this.getTotalTimeElapsed(events));
    }

    @Override
    public void checkSessionIsNotPaused(ExploratoryExecutionRunningState currentState) {
        boolean isPaused = ExploratoryExecutionRunningState.PAUSED.equals((Object)currentState);
        if (isPaused) {
            throw new IllegalActionForPausedExploratoryExecutionException();
        }
    }

    @Override
    public void checkSessionIsNotStopped(ExploratoryExecutionRunningState currentState) {
        boolean isStopped = ExploratoryExecutionRunningState.STOPPED.equals((Object)currentState);
        if (isStopped) {
            throw new IllegalActionForStoppedExploratoryExecutionException();
        }
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'EXECUTE')  or hasRole('ROLE_ADMIN')")
    @CheckBlockingMilestone(entityType=Execution.class)
    public void assignUser(@Id Long executionId, Long userId) {
        this.checkSprintStatusAllowModification(executionId);
        ExploratoryExecution exploratoryExecution = (ExploratoryExecution)this.entityManager.find(ExploratoryExecution.class, (Object)executionId);
        Optional user = this.userDao.findById(userId);
        if (user.isPresent()) {
            exploratoryExecution.setAssigneeUser((User)user.get());
        } else {
            exploratoryExecution.setAssigneeUser(null);
        }
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'EXECUTE')  or hasRole('ROLE_ADMIN')")
    @CheckBlockingMilestone(entityType=Execution.class)
    public void updateTaskDivision(@Id Long executionId, String taskDivision) {
        this.checkSprintStatusAllowModification(executionId);
        ExploratoryExecution exploratoryExecution = (ExploratoryExecution)this.entityManager.find(ExploratoryExecution.class, (Object)executionId);
        if (taskDivision.isEmpty()) {
            exploratoryExecution.setTaskDivision(null);
        } else {
            exploratoryExecution.setTaskDivision(taskDivision);
        }
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'EXECUTE')  or hasRole('ROLE_ADMIN')")
    @CheckBlockingMilestone(entityType=Execution.class)
    public void updateReviewStatus(@Id Long executionId, boolean reviewed) {
        this.checkSprintStatusAllowModification(executionId);
        ExploratoryExecution exploratoryExecution = (ExploratoryExecution)this.entityManager.find(ExploratoryExecution.class, (Object)executionId);
        exploratoryExecution.setReviewed(reviewed);
    }

    @Override
    @PreAuthorize(value="hasPermission(#executionId, 'org.squashtest.tm.domain.execution.Execution', 'READ') or hasRole('ROLE_ADMIN')")
    public boolean isExecutionRunning(@Id Long executionId) {
        return ExploratoryExecutionRunningState.RUNNING.equals((Object)this.findExploratoryExecutionRunningState(executionId));
    }

    @Override
    public void pauseRunningExploratoryExecutions(Long sprintId) {
        this.exploratoryExecutionDao.findAllExploratoryExecutionsBySprintId(sprintId).stream().filter(execId -> this.findExploratoryExecutionRunningState((long)execId).equals((Object)ExploratoryExecutionRunningState.RUNNING)).forEach(execId -> {
            this.createEvent((long)execId, UserContextHolder.getUsername(), ExploratoryExecutionEventType.PAUSE);
            this.updateExecutionMetadata((long)execId);
        });
    }

    private int getTotalTimeElapsed(List<ExploratoryExecutionEvent> events) {
        int totalTimeElapsed = 0;
        int i = 1;
        while (i < events.size()) {
            ExploratoryExecutionEvent previousEvent;
            ExploratoryExecutionEvent currentEvent = events.get(i);
            if (this.isPauseOrStop(currentEvent) && !this.isPause(previousEvent = events.get(i - 1))) {
                if (!this.isStartOrResume(previousEvent)) {
                    throw new IllegalExploratoryExecutionEventSequenceException();
                }
                long difference = currentEvent.getDate().getTime() - previousEvent.getDate().getTime();
                totalTimeElapsed = (int)((long)totalTimeElapsed + difference / 1000L);
            }
            ++i;
        }
        return totalTimeElapsed;
    }

    private boolean isPauseOrStop(ExploratoryExecutionEvent event) {
        return ExploratoryExecutionEventType.PAUSE.equals((Object)event.getEventType()) || ExploratoryExecutionEventType.STOP.equals((Object)event.getEventType());
    }

    private boolean isPause(ExploratoryExecutionEvent event) {
        return ExploratoryExecutionEventType.PAUSE.equals((Object)event.getEventType());
    }

    private boolean isStartOrResume(ExploratoryExecutionEvent event) {
        return ExploratoryExecutionEventType.START.equals((Object)event.getEventType()) || ExploratoryExecutionEventType.RESUME.equals((Object)event.getEventType());
    }

    private void checkSprintStatusAllowModification(long executionId) {
        SprintStatus sprintStatus = this.sprintDisplayDao.getSprintStatusByExecutionId(executionId);
        if (SprintStatus.CLOSED.equals((Object)sprintStatus)) {
            throw new SprintClosedException();
        }
    }
}

