/*
 * Decompiled with CFR 0.152.
 */
package org.squashtest.tm.plugin.scm.git.internal;

import jakarta.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.CreateBranchCommand;
import org.eclipse.jgit.api.FetchCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.ListBranchCommand;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.api.RebaseCommand;
import org.eclipse.jgit.api.RebaseResult;
import org.eclipse.jgit.api.ResetCommand;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.api.errors.InvalidRemoteException;
import org.eclipse.jgit.api.errors.JGitInternalException;
import org.eclipse.jgit.api.errors.RefNotFoundException;
import org.eclipse.jgit.api.errors.TransportException;
import org.eclipse.jgit.diff.DiffEntry;
import org.eclipse.jgit.internal.storage.file.FileRepository;
import org.eclipse.jgit.internal.storage.file.LockFile;
import org.eclipse.jgit.lib.BranchTrackingStatus;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryCache;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.merge.MergeStrategy;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.storage.file.WindowCacheConfig;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RemoteRefUpdate;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.eclipse.jgit.util.FS;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;
import org.squashtest.csp.core.bugtracker.core.UnsupportedAuthenticationModeException;
import org.squashtest.tm.domain.servers.BasicAuthenticationCredentials;
import org.squashtest.tm.domain.servers.Credentials;
import org.squashtest.tm.domain.servers.TokenAuthCredentials;
import org.squashtest.tm.plugin.scm.git.internal.GitClient;
import org.squashtest.tm.plugin.scm.git.internal.exception.GitPluginException;
import org.squashtest.tm.plugin.scm.git.internal.exception.InvalidRepositoryPathException;
import org.squashtest.tm.plugin.scm.git.internal.exception.NoSuchFolderException;
import org.squashtest.tm.plugin.scm.git.internal.exception.NoSuchGitBranchException;
import org.squashtest.tm.plugin.scm.git.internal.exception.NotEmptyDirectoryException;
import org.squashtest.tm.plugin.scm.git.internal.exception.NotGitRepositoryException;
import org.squashtest.tm.plugin.scm.git.internal.util.FileHelper;

@Component
@Scope(value="prototype")
public class GitClientImpl
implements GitClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(GitClientImpl.class);
    private static final String SLASH_DOT_GIT = "/.git";
    private static final String SLASH_HEAD = "/HEAD";
    private static final String SLASH_REFS_SLASH_HEADS = "/refs/heads/";
    private static final String REPOSITORY_NAME = "repositoryName";
    private static final String TOKEN_USERNAME = "<Token>";
    @Inject
    private MessageSource i18nHelper;
    private Git git;
    private FileRepository repository;

    @Override
    public void closeRepository() {
        this.repository.close();
        WindowCacheConfig config = new WindowCacheConfig();
        config.install();
    }

    private String getMessage(String i18nKey) {
        Locale locale = LocaleContextHolder.getLocale();
        return this.i18nHelper.getMessage(i18nKey, null, locale);
    }

    private String getMessage(String i18nKey, Object[] args) {
        Locale locale = LocaleContextHolder.getLocale();
        return this.i18nHelper.getMessage(i18nKey, args, locale);
    }

    @Override
    public void createRepository(String remoteRepositoryUri, String localGitRepositoryPath, String workingBranch, Credentials credentials) throws GitPluginException {
        this.checkDirectoryValidityBeforeCloning(localGitRepositoryPath);
        this.git = this.clone(remoteRepositoryUri, localGitRepositoryPath, workingBranch, credentials);
        this.repository = (FileRepository)this.git.getRepository();
    }

    private void checkDirectoryValidityBeforeCloning(String localRepositoryDirectoryPath) {
        try {
            Paths.get(localRepositoryDirectoryPath, new String[0]);
        }
        catch (InvalidPathException ex) {
            throw new InvalidRepositoryPathException(this.getMessage("error.message.invalidRepositoryPath"), ex);
        }
        if (!(!FileHelper.isFileExist(localRepositoryDirectoryPath) || FileHelper.isDirectory(localRepositoryDirectoryPath) && FileHelper.isEmpty(localRepositoryDirectoryPath))) {
            throw new NotEmptyDirectoryException(this.getMessage("error.message.notEmptyRepositoryPath"));
        }
    }

    public void loadRepository(String localGitDirectoryPath) throws InvalidRepositoryPathException, NoSuchFolderException, NotGitRepositoryException {
        this.checkRepositoryValidityBeforeLoading(localGitDirectoryPath);
        this.repository = this.doLoadLocalRepository(localGitDirectoryPath);
        this.disableSymlinks((Repository)this.repository);
        this.git = new Git((Repository)this.repository);
    }

    private void disableSymlinks(Repository repository) {
        try {
            StoredConfig config = repository.getConfig();
            config.setString("core", null, "symlinks", "false");
            config.save();
        }
        catch (IOException ex) {
            throw new GitPluginException(this.getMessage("error.message.saveConfig"), ex);
        }
    }

    private void checkRepositoryValidityBeforeLoading(String localGitDirectoryPath) {
        LOGGER.trace("Searching folder at path '{}'.", (Object)localGitDirectoryPath);
        try {
            Paths.get(localGitDirectoryPath, new String[0]);
        }
        catch (InvalidPathException ex) {
            throw new InvalidRepositoryPathException(this.getMessage("error.message.invalidRepositoryPath"), ex);
        }
        if (!FileHelper.isFileExist(localGitDirectoryPath) || !FileHelper.isDirectory(localGitDirectoryPath)) {
            throw new NoSuchFolderException(this.getMessage("error.message.folderDoesNotExist", new Object[]{localGitDirectoryPath}));
        }
        LOGGER.debug("Successfully found the folder at path '{}'.", (Object)localGitDirectoryPath);
        LOGGER.trace("Identifying the git repository at path '{}'.", (Object)localGitDirectoryPath);
        File localGitFolder = new File(localGitDirectoryPath);
        if (!RepositoryCache.FileKey.isGitRepository((File)localGitFolder, (FS)FS.DETECTED)) {
            throw new NotGitRepositoryException(this.getMessage("error.message.invalidGitRepository", new Object[]{localGitDirectoryPath}));
        }
        LOGGER.debug("Successfully identified the git repository at path '{}'.", (Object)localGitDirectoryPath);
    }

    public FileRepository doLoadLocalRepository(String localGitDirectoryPath) {
        LOGGER.debug("Loading the local git repository at path '{}'.", (Object)localGitDirectoryPath);
        try {
            return new FileRepository(localGitDirectoryPath);
        }
        catch (IOException ioEx) {
            throw new GitPluginException(this.getMessage("error.message.loadLocalRepository", new Object[]{localGitDirectoryPath}), ioEx);
        }
    }

    @Override
    public void initClientIfNeeded(String localGitDirectoryPath) {
        if (this.repository == null) {
            try {
                LOGGER.debug("Git repository was not loaded. Initializing the GitClient.");
                this.loadRepository(localGitDirectoryPath + SLASH_DOT_GIT);
            }
            catch (GitPluginException ex) {
                throw new GitPluginException(this.getMessage("error.message.initializeGitClient"), (Throwable)((Object)ex));
            }
        } else {
            LOGGER.trace("GitClient is already initialized.");
        }
    }

    public Git clone(String remoteGitRepositoryUri, String localGitRepositoryPath, String workingBranch, Credentials credentials) {
        String username = this.getCredentialUsername(credentials);
        LOGGER.debug("Cloning the remote repository '{}' in the directory '{}' with username '{}'.", new Object[]{remoteGitRepositoryUri, localGitRepositoryPath, username});
        File localGitRepositoryFile = new File(localGitRepositoryPath);
        try {
            Git gitInstance = ((CloneCommand)Git.cloneRepository().setURI(remoteGitRepositoryUri).setDirectory(localGitRepositoryFile).setCredentialsProvider((CredentialsProvider)this.generateCredentialsProvider(credentials))).setBranch(workingBranch).call();
            this.removeSymlinks(workingBranch, credentials, gitInstance, localGitRepositoryFile);
            return gitInstance;
        }
        catch (TransportException ex) {
            if (ex.getMessage().matches("Remote branch '.*' not found in upstream origin")) {
                throw this.buildBranchNotFoundException(workingBranch, (Exception)((Object)ex));
            }
            throw new GitPluginException(this.getMessage("error.message.cloneRepository", new Object[]{remoteGitRepositoryUri}) + "\n" + this.getMessage("error.message.transportError"), ex, REPOSITORY_NAME);
        }
        catch (InvalidRemoteException ex) {
            throw new GitPluginException(this.getMessage("error.message.cloneRepository", new Object[]{remoteGitRepositoryUri}) + "\n" + this.getMessage("error.message.invalidRepositoryUrl"), ex, REPOSITORY_NAME);
        }
        catch (RefNotFoundException ex) {
            throw this.buildBranchNotFoundException(workingBranch, (Exception)((Object)ex));
        }
        catch (IOException | GitAPIException | JGitInternalException ex) {
            throw new GitPluginException(this.getMessage("error.message.cloneRepository", new Object[]{remoteGitRepositoryUri}) + "\n" + this.getMessage("error.message.generalError"), ex, REPOSITORY_NAME);
        }
    }

    private GitPluginException buildBranchNotFoundException(String branchName, Exception ex) {
        return new GitPluginException(this.getMessage("error.message.branchNotFound", new Object[]{branchName}), ex, "branch");
    }

    private void removeSymlinks(String workingBranch, Credentials credentials, Git gitInstance, File localGitRepositoryFile) throws IOException, GitAPIException {
        this.disableSymlinks(gitInstance.getRepository());
        this.deleteWorkingDirectoryContent(localGitRepositoryFile);
        ((FetchCommand)gitInstance.fetch().setCredentialsProvider((CredentialsProvider)this.generateCredentialsProvider(credentials))).setRemote("origin").call();
        if (workingBranch != null) {
            gitInstance.checkout().setName(workingBranch).call();
        }
    }

    private void deleteWorkingDirectoryContent(File repositoryDirectory) throws IOException {
        File[] files = repositoryDirectory.listFiles();
        if (files != null) {
            File[] fileArray = files;
            int n = files.length;
            int n2 = 0;
            while (n2 < n) {
                File file = fileArray[n2];
                if (!".git".equals(file.getName())) {
                    this.deleteRecursively(file);
                }
                ++n2;
            }
        }
    }

    private void deleteRecursively(File file) throws IOException {
        File[] files;
        if (file.isDirectory() && (files = file.listFiles()) != null) {
            File[] fileArray = files;
            int n = files.length;
            int n2 = 0;
            while (n2 < n) {
                File f = fileArray[n2];
                this.deleteRecursively(f);
                ++n2;
            }
        }
        Files.delete(file.toPath());
    }

    @Override
    public void prepareRepository(String workingBranch, String committerMail, Credentials credentials) {
        LOGGER.trace("Preparing repository.");
        this.removeLockFiles(workingBranch);
        this.commit("SquashTM", committerMail, "Commit of remaining non-committed files.");
        this.revertUncommittedChanges();
        this.manageBranchSwitching(workingBranch, credentials);
        this.fetch(credentials);
        this.rebase();
        LOGGER.trace("Successfully prepared repository.");
    }

    public void removeLockFiles(String branchName) {
        if (!LockFile.unlock((File)this.getIndexFile())) {
            throw new GitPluginException(this.getMessage("error.message.removeIndexLockfile"));
        }
        if (!LockFile.unlock((File)this.getHeadFile())) {
            throw new GitPluginException(this.getMessage("error.message.removeHeadLockfile"));
        }
        String formerBranch = this.getCurrentBranchName();
        if (!LockFile.unlock((File)this.getBranchFile(formerBranch))) {
            throw new GitPluginException(this.getMessage("error.message.removeFormerBranchLockfile", new Object[]{formerBranch}));
        }
        if (!formerBranch.equals(branchName) && !LockFile.unlock((File)this.getBranchFile(branchName))) {
            throw new GitPluginException(this.getMessage("error.message.removeNewBranchLockfile", new Object[]{branchName}));
        }
    }

    public File getIndexFile() {
        return this.git.getRepository().getIndexFile();
    }

    private File getHeadFile() {
        return new File(String.valueOf(this.repository.getDirectory().getAbsoluteFile()) + SLASH_HEAD);
    }

    public String getCurrentBranchName() {
        try {
            return this.git.getRepository().getBranch();
        }
        catch (IOException ioe) {
            throw new GitPluginException(this.getMessage("error.message.getCurrentBranchName"), ioe);
        }
    }

    public File getBranchFile(String branchName) {
        return new File(this.repository.getDirectory().getAbsolutePath() + SLASH_REFS_SLASH_HEADS + branchName);
    }

    public void revertUncommittedChanges() {
        LOGGER.trace("Reverting uncommitted changes.");
        try {
            this.git.clean().setForce(true).call();
            this.git.reset().setMode(ResetCommand.ResetType.HARD).call();
            LOGGER.debug("Successfully reverted uncommitted changes.");
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.revertUncommittedChanges"), ex);
        }
    }

    public void manageBranchSwitching(String branchNameParam, Credentials credentials) throws NoSuchGitBranchException {
        LOGGER.trace("Managing branch switch.");
        String currentBranch = this.getCurrentBranchName();
        LOGGER.trace("Current git branch is '{}'.", (Object)currentBranch);
        if (currentBranch != null && currentBranch.equals(branchNameParam)) {
            LOGGER.debug("The current branch IS the right one.");
            return;
        }
        LOGGER.debug("The current branch is NOT the right one.");
        this.revertUncommittedChanges();
        List<Ref> localBranches = this.getLocalBranchesList();
        if (this.containsLocalBranch(localBranches, branchNameParam)) {
            this.switchBranch(branchNameParam);
            return;
        }
        this.fetch(credentials);
        List<Ref> remoteBranches = this.getRemoteBranchesList();
        String remoteBranchFullName = this.findBranchAmongRemoteBranches(remoteBranches, branchNameParam);
        if (remoteBranchFullName != null) {
            this.switchToNewBranch(branchNameParam, remoteBranchFullName);
            return;
        }
        throw new NoSuchGitBranchException(this.getMessage("error.message.branchDoesNotExist", new Object[]{branchNameParam}));
    }

    public List<Ref> getLocalBranchesList() {
        try {
            LOGGER.debug("Listing local git branches.");
            return this.git.branchList().call();
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.listLocalBranches"), ex);
        }
    }

    private boolean containsLocalBranch(List<Ref> localBranchListParam, String testBranchNameParam) {
        for (Ref ref : localBranchListParam) {
            String[] array = ref.getName().split("refs/heads/");
            String branchName = array[1];
            if (!testBranchNameParam.equals(branchName)) continue;
            LOGGER.debug("Branch '{}' DOES exist.", (Object)testBranchNameParam);
            return true;
        }
        LOGGER.debug("Branch '{}' DOES NOT exist.", (Object)testBranchNameParam);
        return false;
    }

    public void switchBranch(String branchNameParam) {
        try {
            LOGGER.debug("Checking out the branch '{}'.", (Object)branchNameParam);
            this.git.checkout().setName(branchNameParam).call();
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.switchToBranch", new Object[]{branchNameParam}), ex);
        }
    }

    public void fetch(Credentials credentials) {
        LOGGER.trace("Fetching remote git repository.");
        try {
            ((FetchCommand)this.git.fetch().setCredentialsProvider((CredentialsProvider)this.generateCredentialsProvider(credentials))).call();
            LOGGER.debug("Successfully fetched the remote git repository.");
        }
        catch (TransportException ex) {
            throw new GitPluginException(this.getMessage("error.message.fetchError") + "\n" + this.getMessage("error.message.transportError"), ex, REPOSITORY_NAME);
        }
        catch (InvalidRemoteException | JGitInternalException ex) {
            throw new GitPluginException(this.getMessage("error.message.fetchError") + "\n" + this.getMessage("error.message.invalidRepositoryUrl"), ex, REPOSITORY_NAME);
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.fetchError") + "\n" + this.getMessage("error.message.generalError"), ex, REPOSITORY_NAME);
        }
    }

    @Override
    public void testCredentials(Credentials credentials) {
        try {
            ((PushCommand)this.git.push().setCredentialsProvider((CredentialsProvider)this.generateCredentialsProvider(credentials))).setDryRun(true).call();
        }
        catch (TransportException ex) {
            throw new GitPluginException(this.getMessage("error.message.connectRepository") + "\n" + this.getMessage("error.message.transportError") + "\n" + this.getMessage("error.message.contactAdmin"), ex);
        }
        catch (InvalidRemoteException | JGitInternalException ex) {
            throw new GitPluginException(this.getMessage("error.message.initializeGitClient") + "\n" + this.getMessage("error.message.contactAdminIfProblemPersist"), ex);
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.connectRepository") + "\n" + this.getMessage("error.message.contactAdmin"), ex);
        }
    }

    private UsernamePasswordCredentialsProvider generateCredentialsProvider(Credentials credentials) {
        switch (credentials.getImplementedProtocol()) {
            case BASIC_AUTH: {
                BasicAuthenticationCredentials basicCredentials = (BasicAuthenticationCredentials)credentials;
                return new UsernamePasswordCredentialsProvider(basicCredentials.getUsername(), basicCredentials.getPassword());
            }
            case TOKEN_AUTH: {
                TokenAuthCredentials tokenCredentials = (TokenAuthCredentials)credentials;
                return new UsernamePasswordCredentialsProvider(TOKEN_USERNAME, tokenCredentials.getToken());
            }
        }
        throw new UnsupportedAuthenticationModeException(credentials.getImplementedProtocol().toString());
    }

    public List<Ref> getRemoteBranchesList() {
        try {
            LOGGER.debug("Listing remote git branches.");
            return this.git.branchList().setListMode(ListBranchCommand.ListMode.REMOTE).call();
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.listRemoteBranches"), ex);
        }
    }

    private String findBranchAmongRemoteBranches(List<Ref> remoteBranchListParam, String testBranchNameParam) {
        for (Ref ref : remoteBranchListParam) {
            String[] array = ref.getName().split("refs/remotes/origin/");
            String branchName = array[1];
            if (!testBranchNameParam.equals(branchName)) continue;
            LOGGER.debug("Branch '{}' DOES exist.", (Object)testBranchNameParam);
            return ref.getName();
        }
        LOGGER.debug("Branch '{}' DOES NOT exist.", (Object)testBranchNameParam);
        return null;
    }

    public String getRemoteRepositoryUrl() {
        return this.repository.getConfig().getString("remote", "origin", "url");
    }

    public void switchToNewBranch(String branchNameParam, String remoteBranchName) {
        try {
            LOGGER.debug("Creating and checking out branch '{}', setting upstream to branch '{}'.", (Object)branchNameParam, (Object)remoteBranchName);
            this.git.checkout().setCreateBranch(true).setName(branchNameParam).setUpstreamMode(CreateBranchCommand.SetupUpstreamMode.TRACK).setStartPoint(remoteBranchName).call();
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.switchToNewBranchAndSetUpstreamToBranch", new Object[]{branchNameParam, remoteBranchName}), ex);
        }
    }

    @Override
    public void rebase() {
        LOGGER.trace("Pulling and rebasing on the remote branch.");
        try {
            BranchTrackingStatus status = this.getBranchTrackingStatus();
            if (status == null) {
                throw new GitPluginException(this.getMessage("error.message.noRemoteRepositoryConfigured"));
            }
            int numberOfCommitsAhead = status.getAheadCount();
            int numberOfCommitsBehind = status.getBehindCount();
            String remoteTrackingBranch = status.getRemoteTrackingBranch();
            if (numberOfCommitsBehind != 0) {
                RebaseResult rebaseResult = this.git.rebase().setUpstream(remoteTrackingBranch).setStrategy((MergeStrategy)MergeStrategy.RECURSIVE).call();
                if (rebaseResult.getStatus().isSuccessful()) {
                    LOGGER.debug("Successfully pulled and rebased on the remote branch.");
                } else {
                    int rebaseTry = 0;
                    while (rebaseResult.getStatus() != RebaseResult.Status.OK) {
                        LOGGER.debug("There is a conflict in the rebase. Trying to resolve it.");
                        if (rebaseTry > numberOfCommitsAhead) {
                            LOGGER.warn("Could not resolve conflict during rebase. Aborting operation.");
                            this.git.rebase().setOperation(RebaseCommand.Operation.ABORT).call();
                            throw new GitPluginException(this.getMessage("error.message.rebaseAbort"));
                        }
                        RevCommit conflictCommit = rebaseResult.getCurrentCommit();
                        Set conflictsSet = this.git.status().call().getConflicting();
                        ArrayList conflictsList = new ArrayList(conflictsSet);
                        this.git.checkout().addPaths(conflictsList).setStartPoint(conflictCommit).call();
                        for (String path : conflictsList) {
                            this.git.add().addFilepattern(path).call();
                        }
                        rebaseResult = this.git.rebase().setOperation(RebaseCommand.Operation.CONTINUE).call();
                        ++rebaseTry;
                    }
                }
            } else {
                LOGGER.debug("No commits were made on remote repository. No need to Pull and Rebase.");
            }
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.pullRebaseOnRemoteBranch"), ex);
        }
    }

    public BranchTrackingStatus getBranchTrackingStatus() {
        try {
            return BranchTrackingStatus.of((Repository)this.repository, (String)this.repository.getBranch());
        }
        catch (IOException ioEx) {
            throw new GitPluginException(this.getMessage("error.message.getBranchTrackingStatus"), ioEx);
        }
    }

    @Override
    public void commit(String committerNameParam, String committerMailParam, String commitMessageParam) {
        try {
            if (!this.diffInIndex().isEmpty()) {
                LOGGER.trace("Committing with author '{}' and message '{}'.", (Object)committerNameParam, (Object)commitMessageParam);
                this.git.commit().setCommitter(committerNameParam, committerMailParam).setMessage(commitMessageParam).call();
                LOGGER.debug("Successfully committed with committer '{}' and message '{}'.", (Object)committerNameParam, (Object)commitMessageParam);
            } else {
                LOGGER.debug("No need to commit because there are no modifications in the index.");
            }
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.commitWithCommitterAndMessage", new Object[]{committerNameParam, commitMessageParam}), ex);
        }
    }

    public List<DiffEntry> diffInIndex() {
        try {
            return this.git.diff().setCached(true).call();
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.getDiffEntriesInGitIndex"), ex);
        }
    }

    @Override
    public void stageFilesToIndex(Path relativePath) {
        this.addFilesToIndex(relativePath);
        this.addDeletedFilesToIndex(relativePath);
    }

    public void addFilesToIndex(Path fileRelativePath) {
        String path = fileRelativePath.toString();
        path = path.isEmpty() ? "." : path.trim().replace("\\", "/");
        try {
            LOGGER.debug("Adding file(s) with path '{}' to the git index.", (Object)path);
            this.git.add().addFilepattern(path).call();
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.addFilesToIndex"), ex);
        }
    }

    public void addDeletedFilesToIndex(Path filesRelativePath) {
        String path = filesRelativePath.toString();
        path = path.isEmpty() ? "" : path.trim().replace("\\", "/");
        try {
            LOGGER.debug("Adding deletion of file(s) with path {} to the git index.", (Object)path);
            for (DiffEntry entry : this.git.diff().call()) {
                if (!entry.getChangeType().equals((Object)DiffEntry.ChangeType.DELETE) || !entry.getOldPath().startsWith(path)) continue;
                this.deleteFilesAndAddToIndex(Paths.get(entry.getOldPath(), new String[0]));
            }
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.addDeletedFilesToIndex"), ex);
        }
    }

    public void deleteFilesAndAddToIndex(Path filesRelativePath) {
        String path = filesRelativePath.toString();
        path = path.trim().replace("\\", "/");
        try {
            LOGGER.debug("Deleting file(s) with path {} and adding deletion to the git index.", (Object)path);
            this.git.rm().addFilepattern(path).call();
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.deleteFilesAndAddToIndex"), ex);
        }
    }

    @Override
    public boolean safePush(Credentials credentials) {
        this.fetch(credentials);
        BranchTrackingStatus status = this.getBranchTrackingStatus();
        if (status == null) {
            throw new GitPluginException(this.getMessage("error.message.noRemoteRepositoryConfigured"));
        }
        int ahead = status.getAheadCount();
        LOGGER.debug("Current branch is ahead of remote branch by {} commit(s).", (Object)ahead);
        int behind = status.getBehindCount();
        LOGGER.debug("Current branch is behind remote branch by {} commit(s).", (Object)behind);
        if (behind > 0) {
            LOGGER.debug("Can not push for the moment because local branch is behind remote branch by {} commit(s).", (Object)behind);
            return false;
        }
        if (ahead > 0) {
            LOGGER.debug("There are {} commit(s) ahead to push.", (Object)ahead);
            this.push(credentials);
            return true;
        }
        LOGGER.debug("Did not push because there are no commits ahead.");
        return true;
    }

    public void push(Credentials credentials) {
        String username = this.getCredentialUsername(credentials);
        try {
            LOGGER.trace("Pushing commit(s) with username '{}'.", (Object)username);
            Iterable pushResults = ((PushCommand)this.git.push().setCredentialsProvider((CredentialsProvider)this.generateCredentialsProvider(credentials))).call();
            for (PushResult pushResult : pushResults) {
                for (RemoteRefUpdate remoteRefUpdate : pushResult.getRemoteUpdates()) {
                    if (remoteRefUpdate.getStatus() == RemoteRefUpdate.Status.OK) continue;
                    throw new GitPluginException(this.getMessage("error.message.pushWithUsernameWithMessage", new Object[]{username, pushResult.getMessages()}));
                }
            }
            LOGGER.debug("Successfully pushed commit(s) with username '{}'.", (Object)username);
        }
        catch (GitAPIException ex) {
            throw new GitPluginException(this.getMessage("error.message.pushWithUsername", new Object[]{username}), ex);
        }
    }

    private String getCredentialUsername(Credentials credentials) {
        switch (credentials.getImplementedProtocol()) {
            case BASIC_AUTH: {
                BasicAuthenticationCredentials basicCredentials = (BasicAuthenticationCredentials)credentials;
                return basicCredentials.getUsername();
            }
            case TOKEN_AUTH: {
                return TOKEN_USERNAME;
            }
        }
        throw new UnsupportedAuthenticationModeException(credentials.getImplementedProtocol().toString());
    }
}

