/**
 *     This file is part of the Squashtest platform.
 *     Copyright (C) 2010 - 2011 Squashtest TM, Squashtest.org
 *
 *     See the NOTICE file distributed with this work for additional
 *     information regarding copyright ownership.
 *
 *     This is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     this software is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.squashtest.csp.tm.service;

import java.util.List;

import org.squashtest.csp.tm.domain.library.Folder;
import org.squashtest.csp.tm.domain.library.Library;
import org.squashtest.csp.tm.domain.library.LibraryNode;
import org.squashtest.csp.tm.service.deletion.SuppressionPreviewReport;

/**
 * Defines common methods for a library navigation service, mainly library access and folder manipulation.
 * 
 * @author Gregory Fouquet
 * 
 */
public interface LibraryNavigationService<LIBRARY extends Library<? extends NODE>, FOLDER extends Folder<? extends NODE>, NODE extends LibraryNode> {


	/**
	 * will create a deep copy of the given LibraryNodes, paste them 
	 * in the target folder, and return the copies.
	 * 
	 * 
	 * @param destinationId the id of the folder where you need to copy to.
	 * @param targetId the list of the librarynodes we want copies of.
	 * @return the list of the copies themselves.
	 */
	List<NODE> copyNodesToFolder(long destinationId, Long[] targetId);
	
	/**
	 * same, when the destination is a Library.
	 * 
	 * 
	 * @param destinationId the id of the library where you need to copy to.
	 * @param targetId the list of the librarynodes we want copies of.
	 * @return the list of the copies themselves.
	 */
	List<NODE> copyNodesToLibrary(long destinationId, Long[] targetId);
	
	
	void modeNodesToFolder(long destinationId, Long[] targetId);
	
	void moveNodesToLibrary(long destinationId, Long[] targetId);
	


	void addFolderToLibrary(long destinationId, FOLDER newFolder);

	void addFolderToFolder(long destinationId, FOLDER newFolder);

	FOLDER findFolder(long folderId);


	List<NODE> findLibraryRootContent(long libraryId);

	/**
	 * Returns the content of the folder designated by its id.
	 * 
	 * @param folderId
	 * @return
	 */
	List<NODE> findFolderContent(long folderId);


	@Deprecated
	void renameFolder(long folderId, String newName);


	LIBRARY findLibrary(long libraryId);
	
	
	/**
	 * that method should investigate the consequences of the deletion request, and return a report
	 * about what will happen.
	 * 
	 * @param targetIds
	 * @return
	 */
	List<SuppressionPreviewReport> simulateDeletion(List<Long> targetIds);
	
	/**
	 * that method should delete the nodes. It still takes care of non deletable nodes so
	 * the implementation should filter out the ids who can't be deleted.
	 * 
	 * 
	 * @param targetIds
	 * @return
	 */
	List<Long> deleteNodes(List<Long> targetIds);
	
	

}