/**
 *     This file is part of the Squashtest platform.
 *     Copyright (C) 2010 - 2011 Squashtest TM, Squashtest.org
 *
 *     See the NOTICE file distributed with this work for additional
 *     information regarding copyright ownership.
 *
 *     This is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     this software is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.squashtest.csp.tm.internal.service;

import java.io.InputStream;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import javax.inject.Inject;

import org.springframework.stereotype.Service;
import org.squashtest.csp.tm.domain.attachment.Attachment;
import org.squashtest.csp.tm.domain.attachment.AttachmentContent;
import org.squashtest.csp.tm.domain.attachment.AttachmentList;
import org.squashtest.csp.tm.infrastructure.filter.CollectionSorting;
import org.squashtest.csp.tm.infrastructure.filter.FilteredCollectionHolder;
import org.squashtest.csp.tm.internal.repository.AttachmentContentDao;
import org.squashtest.csp.tm.internal.repository.AttachmentDao;
import org.squashtest.csp.tm.internal.repository.AttachmentListDao;
import org.squashtest.csp.tm.service.AttachmentManagerService;

/*
 * FIXME !
 *
 * we can't secure the operations on attachments yet, because they delegate the security permissions
 * to the owning entity. The problem being that the ownership of an AttachmentList is one way : from the
 * entity to the list.
 *
 * Therefore we can't perform a permission check using the AttachmentList id alone, one would need to fetch
 * the owning entity back first. That would require additional work Dao-side.
 *
 * See task #102 on ci.squashtest.org/mantis
 *
 */
@Service("squashtest.tm.service.AttachmentManagerService")
public class AttachmentManagerServiceImpl implements AttachmentManagerService {

	@Inject
	private AttachmentDao attachmentDao;

	@Inject
	private AttachmentContentDao contentDao;

	@Inject
	private AttachmentListDao attachmentListDao;

	@Override
	public Long addAttachment(Long attachmentListId, Attachment attachment) {
		attachment.setAddedOn(new Date());
		AttachmentList list = attachmentListDao.findById(attachmentListId);

		list.addAttachment(attachment);

		// check first if content is provided along.
		AttachmentContent content = attachment.getContent();
		if (content != null) {
			contentDao.persist(content);
		}

		attachmentDao.persist(attachment);
		return attachment.getId();
	}

	@Override
	public Attachment findAttachment(Long attachmentId) {
		return attachmentDao.findById(attachmentId);
	}

	@Override
	public Set<Attachment> findAttachments(Long attachmentListId) {
		return attachmentDao.findAllAttachments(attachmentListId);
	}

	@Override
	public void setAttachmentContent(InputStream stream, Long attachmentId) {
		Attachment attachment = attachmentDao.findAttachmentWithContent(attachmentId);

		AttachmentContent content = attachment.getContent();

		if (content == null) {
			content = new AttachmentContent();
			content.setContent(stream);
			contentDao.persist(content);
			attachment.setContent(content);
		}

		content.setContent(stream);
	}

	@Override
	public InputStream getAttachmentContent(Long attachmentId) {
		Attachment attachment = attachmentDao.findAttachmentWithContent(attachmentId);
		return attachment.getContent().getContent();
	}

	@Override
	public void removeAttachmentFromList(Long attachmentListId, Long attachmentId) {
		AttachmentList list = attachmentListDao.findById(attachmentListId);
		Attachment attachment = attachmentDao.findById(attachmentId);

		list.removeAttachment(attachment);
		attachmentDao.removeAttachment(attachment.getId());
	}

	@Override
	public void removeListOfAttachments(Long attachmentListId, List<Long> attachmentIds) {

		Iterator<Attachment> iterAttach = attachmentListDao.findById(attachmentListId).getAllAttachments().iterator();

		while (iterAttach.hasNext()) {
			Attachment att = iterAttach.next();
			ListIterator<Long> iterIds = attachmentIds.listIterator();

			while (iterIds.hasNext()) {
				Long id = iterIds.next();
				if (id.equals(att.getId())) {
					iterAttach.remove();
					iterIds.remove();
					attachmentDao.removeAttachment(att.getId());
					break;
				}
			}
			if (attachmentIds.size() == 0) {
				break;
			}
		}
	}

	@Override
	public void renameAttachment(Long attachmentId, String newName) {
		Attachment attachment = attachmentDao.findById(attachmentId);
		attachment.setShortName(newName);
	}

	@Override
	public String findAttachmentShortName(Long attachmentId) {
		Attachment attachment = attachmentDao.findById(attachmentId);
		return attachment.getShortName();
	}

	@Override
	public FilteredCollectionHolder<List<Attachment>> findFilteredAttachmentForList(long attachmentListId,
			CollectionSorting filter) {
		List<Attachment> atts = attachmentDao.findAllAttachmentsFiltered(attachmentListId, filter);
		long count = attachmentDao.findAllAttachments(attachmentListId).size();
		return new FilteredCollectionHolder<List<Attachment>>(count, atts);
	}

}
